import { BundleData } from 'client/bundles/types';
import cn from 'client/utils/cn';
import React, { Component } from 'react';

import DropTouchEvents from 'client/components/helpers/drop-touch-events';

import AdaptiveCountryList from './adaptive-country-list';
import CountryBadge from './country-badge';

import './index.css';

import 'client/components/main-title/index.css';

type Size = [number, number];

const b = cn('map-yandex-geography');
const bMainTitle = cn('main-title');

const MAP_ORIG_SIZE = [1230, 628];

interface Props {
    data: BundleData['main']['geography'];
}

class MapYandexGeography extends Component<Props> {
    static reduceCoordinates([x, y]: Size): Size {
        const [mapWidth, mapHeight] = MAP_ORIG_SIZE;

        return [
            x / mapWidth * 100,
            y / mapHeight * 100
        ];
    }

    badges: {[key in Props['data']['countries'][number]['type']]: string};
    picRef = React.createRef<HTMLDivElement>();

    constructor(props: Props) {
        super(props);

        this.badges = {
            'office': props.data['office-badge'],
            'location': props.data['location-badge']
        };
    }

    renderNoteItem = (text: string, imgUrl: string) => {
        return (
            <div className={b('note-item')}>
                <span
                    className={b('note-img')}
                    style={{ backgroundImage: `url(${imgUrl})` }}
                    />
                { text }
            </div>);
    }

    render() {
        const { countries, note } = this.props.data;

        return (
            <article className={b()}>
                <h2 className={bMainTitle()}>Geography of Operations</h2>

                <DropTouchEvents>
                    <div ref={this.picRef} className={b('map')}>
                        {countries.map(({ country, description, x, y, type }) => (
                            <CountryBadge
                                key={country}
                                country={country}
                                type={type}
                                description={description}
                                badgeStyle={this.badges[type]}
                                coordinates={MapYandexGeography.reduceCoordinates([x, y])}
                                />
                        ))}
                    </div>
                </DropTouchEvents>

                {
                    note ?
                        <div className={b('note')}>
                            { note.office && this.renderNoteItem(note.office, this.props.data['office-badge']) }
                            { note.location && this.renderNoteItem(note.location, this.props.data['location-badge']) }
                        </div> :
                        null
                }

                <div className={b('adaptive-list')}>
                    <AdaptiveCountryList countries={countries} />
                </div>
            </article>
        );
    }
}

export default MapYandexGeography;
