import cn from 'client/utils/cn';

import React, { Component } from 'react';

import 'client/components/date/index.css';
import moment from 'moment';
import './index.css';

import Breadcrumbs from 'client/components/breadcrumbs';
import DocsList from 'client/components/docs-list';

import { PageSecProps, SecFiling } from 'client/bundles/types';

const b = cn('sec-filing');
const bPageTitle = cn('page-title');

interface Props extends Omit<PageSecProps, 'singleFiling'> {
    title: string;
    singleFiling: SecFiling;
    baseUrl: string;
}

class SecFilingView extends Component<Props> {
    renderDetails = () => {
        const { singleFiling } = this.props;

        const docHtml = singleFiling.documentFormatFiles
            .find(doc => doc.format && doc.format === 'html');

        return (
            <div className={b('details-items')}>
                <div className={b('row', { type: 'form' })}>
                    <span className={b('row-key')}>Form</span>
                    <span className={b('row-value')}>
                        {docHtml ?
                            <a className={b('link')} href={docHtml.link} target={'_blank'}>
                                {singleFiling.formType}
                            </a> :
                            singleFiling.formType
                        }
                    </span>
                </div>
                <div className={b('row')}>
                    <span className={b('row-key')}>Filing Date</span>
                    <span className={b('row-value')}>
                        {moment(singleFiling.pubDate).format('MMM DD, YYYY')}
                    </span>
                </div>
                {
                    singleFiling.periodDate ?
                        <div className={b('row')}>
                            <span className={b('row-key')}>Document Date</span>
                            <span className={b('row-value')}>
                                {moment(singleFiling.periodDate).format('MMM DD, YYYY')}
                            </span>
                        </div> :
                        null
                }
                <div className={b('row')}>
                    <span className={b('row-key')}>Form Description</span>
                    <span className={b('row-value')}>{singleFiling.title}</span>
                </div>
                {
                    singleFiling.filingGroup ?
                        <div className={b('row')}>
                            <span className={b('row-key')}>Filing Group</span>
                            <span className={b('row-value')}>{singleFiling.filingGroup}</span>
                        </div> :
                        null
                }
                <div className={b('row')}>
                    <span className={b('row-key')}>Company</span>
                    <span className={b('row-value')}>{singleFiling.company}</span>
                </div>
                <div className={b('row')}>
                    <span className={b('row-key')}>Issuer</span>
                    <span className={b('row-value')}>{singleFiling.issuer}</span>
                </div>
            </div>
        );
    }

    renderDocs = () => {
        const { singleFiling } = this.props;
        const docFormats = ['html', 'pdf', 'convpdf', 'rtf', 'xls', 'xbrl'];

        const docs = singleFiling.documentFormatFiles
            .filter(doc => doc.format && docFormats.includes(doc.format))
            .map(doc => {
                if (doc.format === 'convpdf') {
                    doc.format = 'pdf';
                    doc.description = 'pdf';
                }

                return {
                    ...doc,
                    title: `${doc.format!.toUpperCase()}`
                };
            });

        return (
            <div className={b('docs-list')}>
                <h2 className={b('title')}>Filing Formats</h2>

                <DocsList docs={docs} />
            </div>
        );
    }

    renderXBRLDocs = () => {
        const { singleFiling } = this.props;

        if (!singleFiling.xbrlSequence) {
            return null;
        }

        const docs = singleFiling.xbrlSequence.map(doc => {
            return {
                ...doc,
                title: doc.format || 'XBRL DOCUMENT',
                format: 'xbrl-sequence'
            };
        });

        return (
            <div className={b('docs-list')}>
                <h2 className={b('title')}>XBRL</h2>

                <DocsList docs={docs} />
            </div>
        );
    }

    render() {
        const { title, baseUrl, currentYear, sectionTitle } = this.props;

        return (
            <div className={b()}>
                <Breadcrumbs text={sectionTitle} link={`${baseUrl}?year=${currentYear}`} />

                <h1 className={bPageTitle()}>{title}</h1>

                <div className={b('content')}>
                    <div className={b('details')}>
                        <h2 className={b('title')}>Document Details</h2>

                        {this.renderDetails()}
                    </div>

                    <div className={b('docs')}>
                        {this.renderDocs()}

                        {this.renderXBRLDocs()}
                    </div>
                </div>
            </div>
        );
    }
}

export default SecFilingView;
