import cn from 'client/utils/cn';

import React, { ChangeEvent, Component } from 'react';

import 'client/components/date/index.css';
import moment from 'moment';
import './index.css';

import DocsList from 'client/components/docs-list';

import { FilingType, PageSecProps, SecFiling } from 'client/bundles/types';

import { AutoscrollerStoragePrefix } from 'client/components/helpers/autoscroller';
import Select, { Option } from 'client/components/lego/select';
import MenuYears from 'client/components/menu-years';

const b = cn('sec-filings');
const bDate = cn('date');
const bPageTitle = cn('page-title');

interface Props extends PageSecProps {
    title: string;
    baseUrl: string;
}

class SecFilings extends Component<Props> {
    onSelectTypesChange = (event: ChangeEvent<HTMLSelectElement>): void => {
        this.requestFilteredData('type', event.target.value);
    }

    requestFilteredData = (paramName: 'type', value: string): void => {
        const url = new URL(window.location.href);

        url.searchParams.set(paramName, value);
        window.document.location.href = url.href;
    }

    getFilingsOptions = (types: FilingType[]): Option[] => {
        return types.map((filing: FilingType) => {
            return {
                key: filing.id,
                value: filing.id,
                content: filing.value
            };
        });
    }

    renderNoFiling = () => {
        return (
            <div className={b('message')}>There are no SEC filings available.</div>
        );
    }

    renderNoData = () => {
        return (
            <div className={b('message')}>
                Something went wrong.<br /><br />
                Try refreshing the page or come back later.
            </div>
        );
    }

    renderTable = (secFilings: SecFiling[]) => {
        return (
            <table className={b('table')}>
                <thead>
                    <tr>
                        <th>Date Filed</th>
                        <th>Description</th>
                        <th>Filing</th>
                    </tr>
                </thead>
                <tbody>
                    { secFilings.map(this.renderRow) }
                </tbody>
            </table>
        );
    }

    renderRow = (item: SecFiling, index: number) => {
        const docFormats = ['html', 'pdf', 'convpdf', 'xls'];

        const docs = item.documentFormatFiles
            .filter(doc => doc.format && docFormats.includes(doc.format))
            .map(doc => {
                if (doc.format === 'convpdf') {
                    doc.format = 'pdf';
                    doc.description = 'pdf';
                }

                return doc;
            });

        return (
            <tr key={index}>
                <td className={b('date')}>
                    <span className={bDate()}>
                        {moment(item.pubDate).format('MMMM DD, YYYY')}
                    </span>
                </td>
                <td className={b('description')}>
                    <h3 className={b('title')}>
                        <a
                            className={b('title-link')}
                            href={`${this.props.baseUrl}?id=${item.guid}`}
                            >
                            {item.title}
                        </a>
                    </h3>

                    <DocsList view={'line'} docs={docs} />
                </td>
                <td className={b('type')}>{item.formType}</td>
            </tr>
        );
    }

    render() {
        const { yearFilings, years, types, currentYear, groupId, title } = this.props;

        return (
            <div className={b()}>
                <h1 className={bPageTitle()}>{title}</h1>

                {years.length > 0 ?
                    <div className={b('content')}>
                        <div className={b('filters')}>
                            {currentYear && years && years.length > 0 &&
                                <MenuYears
                                    currentYear={currentYear}
                                    years={years}
                                    storagePrefix={AutoscrollerStoragePrefix.SecFillings}
                                    />
                            }

                            {types.length > 0 ?
                                <Select
                                    size="m"
                                    theme="normal"
                                    className={b('filings')}
                                    value={groupId}
                                    options={this.getFilingsOptions(types)}
                                    onChange={this.onSelectTypesChange}
                                    /> :
                                null
                            }
                        </div>
                        {yearFilings && yearFilings.length > 0 ?
                            this.renderTable(yearFilings) :
                            this.renderNoFiling()
                        }
                    </div> :
                    this.renderNoData()
                }
            </div>
        );
    }
}

export default SecFilings;
