import cn from 'client/utils/cn';

import React, { ChangeEvent, Component } from 'react';

import './index.css';

import Button from 'client/components/lego/button';
import Textinput from 'client/components/lego/text-input';

const b = cn('subscribe');

interface SubscribeResult {
    status: 'ok' | 'skip';
}

interface SubscribeError {
    statusCode: number;
    body: {
        internalCode: string;
        message: string;
    };
}

interface BlogSubscribeData {
    result?: SubscribeResult;
    error?: SubscribeError;
}

interface State {
    value: string;
    result?: string;
    error?: string;
}

const MESSAGE: Record<string, string> = {
    'empty-message': 'Your email address here',
    'error-valid': 'Please check your email address and try again',
    'error-message': ' Something went wrong. Please try again later ',
    'error-dubl': 'This email address has already been used to subscribe to our news ',
    'ok-message': 'Thank you for signing up to our news. Please check your inbox to confirm your subscription.'
};

class Subscribe extends Component<{}, State> {
    constructor() {
        super({});

        this.state = {
            value: ''
        };
    }

    onChange = (event: ChangeEvent<HTMLInputElement>): void => {
        this.setState({
            value: event.target.value.trim(),
            result: undefined,
            error: undefined
        });
    }

    onSubmit = (): void => {
        const email = this.state.value;

        if (!email) {
            this.setState({ error: MESSAGE['empty-message'] });

            return;
        }

        const emailRegexp = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

        if (!emailRegexp.test(email)) {
            this.setState({ error: MESSAGE['error-valid'] });

            return;
        }

        window.fetch(`/subscribe?email=${email}`)
            .then(response => response.json())
            .then(
                (response: BlogSubscribeData) => {
                    const { result, error } = response;

                    if (result?.status === 'ok') {
                        this.setState({ result: MESSAGE['ok-message'] });

                        return;
                    } else if (result?.status === 'skip') {
                        this.setState({ error: MESSAGE['error-dubl'] });

                        return;
                    }

                    let errorMessageKey = 'error-message';

                    if (error) {
                        const code = error.statusCode;
                        const internalCode = error && error.body && error.body.internalCode;

                        if (code === 403 && internalCode === '403_EAS') {
                            errorMessageKey = 'error-dubl';
                        } else if (code === 400) {
                            errorMessageKey = 'error-valid';
                        }
                    }

                    this.setState({ error: MESSAGE[errorMessageKey] });
                },
                () => this.setState({ error: MESSAGE['error-message'] })
            );
    }

    render() {
        const { error, result, value } = this.state;

        return (
            <div className={b()}>
                <div className={b('title')}>
                    Sign up to our news
                </div>

                <div className={b('form')}>
                    <Textinput
                        theme="normal"
                        size="m"
                        pin="round-clear"
                        hasClear
                        value={value}
                        placeholder="Your email address"
                        onChange={this.onChange}
                        />

                    <Button
                        className={b('button')}
                        size="m"
                        type="submit"
                        theme="action"
                        pin="clear-round"
                        onClick={this.onSubmit}
                        disabled={Boolean(!value)}
                        >
                        Sign up
                    </Button>
                </div>
                {result || error ?
                    <div className={b('result', { error: Boolean(error) })}>
                        {result || error}
                    </div> :
                    null
                }
            </div>
        );
    }
}

export default Subscribe;
