import md5 from 'md5';
import moment from 'moment';

import { NextFunction, Request, Response } from 'express';

export default function eventsRemind(req: Request, res: Response, next: NextFunction) {
    const eventId = req.params.fileName;

    if (!eventId) {
        next();

        return;
    }

    const { bunker } = req;

    const eventsBunkerData = bunker.pages.events;
    const currentYear = new Date().getFullYear();
    const currentYearData = eventsBunkerData && eventsBunkerData[currentYear];

    const currentEventData = (currentYearData || []).find(eventsYearItem =>
        md5(eventsYearItem.dateStart + eventsYearItem.title) === eventId
    );

    if (!currentEventData) {
        next();

        return;
    }

    const uid = `UID:${eventId}`;
    const location = currentEventData.location ? `LOCATION:${currentEventData.location}` : '';
    const summary = `SUMMARY:${currentEventData.title}`;

    const webcastData = (currentEventData.docs || []).find(doc => doc.type === 'webcast');
    const description = webcastData?.link ? `DESCRIPTION:Webcast: ${webcastData.link}` : '';

    const timeStartValue = currentEventData.timeStart?.value;
    const timeEndValue = currentEventData.timeEnd?.value;
    const timeZoneStart = currentEventData.timeStart?.zone;
    const timeZoneEnd = currentEventData.timeEnd?.zone;

    const dtstart = `DTSTART${getTimeZone(timeZoneStart)}:${getFormatDate(currentEventData.dateStart)}${getFormatTime(timeStartValue)}`;
    const dtend = currentEventData.dateEnd ? `DTEND${getTimeZone(timeZoneEnd)}:${getFormatDate(currentEventData.dateEnd)}${getFormatTime(timeEndValue)}` : '';
    const dtstamp = `DTSTAMP:${new Date().getTime()}`;

    const data = [
        'BEGIN:VCALENDAR',
        'VERSION:2.0',
        'PRODID:-//Yandex LLC//Yandex Calendar//EN',
        'CALSCALE:GREGORIAN',
        'BEGIN:VEVENT',
        'ORGANIZER;CN=Yandex N.V.:MAILTO:no-reply@ir.yandex',
        dtstart,
        dtend,
        dtstamp,
        uid,
        location,
        description,
        summary,
        'URL;VALUE=URI:https://ir.yandex/events-and-presentations',
        'END:VEVENT',
        'END:VCALENDAR'
    ].filter(Boolean);

    res.set('Content-Type', 'text/calendar; charset=utf-8');
    res.attachment(`Yandex N.V. - ${currentEventData.title}.ics`);
    res.send(data.join('\n'));
}

function getFormatDate(date: string): string {
    return moment(date).format('YYYYMMDD');
}

function getFormatTime(time: string | undefined): string {
    if (!time) {
        return '';
    }

    return `T${moment(time, 'HH:mm:ss').format('HHmmss')}`;
}

function getTimeZone(zone: string | undefined): string {
    if (!zone) {
        return '';
    }

    if (zone === 'EDT') {
        zone = 'America/New_York';
    }

    return `;TZID=${zone}`;
}
