import { Request, Response } from 'express';
import md5 from 'md5';

import { BunkerEvents, Doc, DocType, EventsYear, OccasionData } from 'client/bundles/types';
import { getActualDocs, getCurrentYear, getYearsWithData } from 'server/utils/bunker-utils';
import { getCommonPageData } from 'server/utils/page-utils';

const DOCS_LIST: Record<DocType, string> = {
    'presentation': 'Presentation',
    'slides': 'Supplementary slides',
    'webcast': 'Webcast'
};

export default function events(req: Request, res: Response) {
    const pageName = 'events';
    const { bunker } = req;

    const years = getYearsWithData(bunker.pages.events);
    const currentYear = getCurrentYear(req, years);
    const yearEvents = preparedEvents(bunker.pages.events, currentYear);
    const commonPageData = getCommonPageData(req, res, pageName);

    res.renderBundle('desktop', pageName, {
        ...commonPageData,
        events: {
            years,
            currentYear,
            title: yearEvents ? yearEvents.title : '',
            yearEvents: yearEvents ? yearEvents.currentYearData : undefined
        }
    });
}

function preparedEvents(eventsData: BunkerEvents | undefined, eventsCurrentYear: string): EventsYear | undefined {
    if (!eventsData) {
        return;
    }

    const sortedEventsData = [...eventsData[eventsCurrentYear]].sort((a, b): number => {
        return new Date(b.dateStart).getTime() - new Date(a.dateStart).getTime();
    });

    const currentYearData = sortedEventsData.map(eventsYearItem => {
        const isFutureDate = new Date(eventsYearItem.dateStart).getTime() > new Date().getTime();

        const remindData = isFutureDate && {
            title: 'Remind me',
            type: 'remind',
            link: `/events-and-presentations/${md5(eventsYearItem.dateStart + eventsYearItem.title)}`,
            format: 'remind'
        };
        const docs = getDocs(eventsYearItem.docs);
        const actualDocs = getActualDocs(docs, eventsYearItem.dateStart);

        return {
            ...eventsYearItem,
            docs: remindData ? [...actualDocs, remindData] : actualDocs
        } as OccasionData;
    });

    return {
        title: eventsData.title as unknown as string,
        currentYearData
    };
}

function getDocs(docs: Doc[]): Doc[] {
    return docs.map(doc => {
        return {
            ...doc,
            title: doc.type ? DOCS_LIST[doc.type] : ''
        };
    });
}
