import {
    BunkerFinancialQuarter,
    BunkerFinancials,
    BunkerFinancialType,
    FinancialsYear,
    REPORT_NAMES
} from 'client/bundles/types';
import { NextFunction, Request, Response } from 'express';
import { getActualDocs, getCurrentYear, getReorderedReports, getYearsWithData } from 'server/utils/bunker-utils';
import { getCommonPageData, getFinancialPageUrl } from 'server/utils/page-utils';

const _ = require('lodash');

/* eslint-disable-next-line max-statements, complexity */
export default function financials(req: Request, res: Response, next: NextFunction) {
    const pageName = 'financials';
    const commonPageData = getCommonPageData(req, res, pageName);
    const { bunker, query: { report: requestedQuarter } } = req;
    let { query: { year: requestedYear } } = req;
    const financialsData = bunker.pages.financials;
    const years = getYearsWithData(financialsData);
    const isReportRequested = requestedQuarter && typeof requestedQuarter === 'string' && REPORT_NAMES.includes(requestedQuarter);
    const isYearData = requestedYear && typeof requestedYear === 'string' && years.includes(requestedYear);

    let yearReports;
    let quarterReport;

    // Если пришли на корень /financial-releases или запросили несуществующий год,
    // нужно показать данные за последний год
    if (!requestedYear || !isYearData) {
        requestedYear = getCurrentYear(req, years);
    }

    if (isReportRequested) {
        if (!isYearData) {
            next();

            return;
        }

        quarterReport = getQuarterReport(financialsData, requestedQuarter as BunkerFinancialType, requestedYear as string);

        if (!quarterReport) {
            next();

            return;
        }

        if (quarterReport && !quarterReport.html && quarterReport.linkPdf) {
            res.redirect(quarterReport.linkPdf);

            return;
        }
    } else {
        yearReports = getYearReports(req, financialsData, requestedYear as string);
    }

    const sectionTitle = financialsData && financialsData.title;

    res.renderBundle('desktop', pageName, {
        ...commonPageData,
        financials: {
            years,
            currentYear: requestedYear,

            title: yearReports ? yearReports.title : sectionTitle,
            yearReports: yearReports ? yearReports.reports : undefined,

            // Запросили отчёт за конкретный квартал.
            quarterReport
        }
    });
}

function getYearReports(req: Request, financialsData: BunkerFinancials | undefined, requestedYear: string): FinancialsYear | undefined {
    if (!financialsData) {
        return;
    }

    const reports = _.cloneDeep(financialsData[requestedYear]);

    Object.keys(reports).forEach(key => {
        reports[key].url = getFinancialPageUrl(req, { year: requestedYear, name: key });
        reports[key].docs = getActualDocs(reports[key].docs, reports[key].date);

        // Для страницы со списком не нужно отдавать поле html.
        // Слишком большое, страница медленно отвечает и падает под нагрузкой.
        reports[key].html = '';
    });

    return {
        title: financialsData.title as unknown as string,
        reports: getReorderedReports(reports)
    };
}

function getQuarterReport(financialsData: BunkerFinancials | undefined, quarter: BunkerFinancialType, requestedYear: string): BunkerFinancialQuarter | undefined {
    if (!financialsData) {
        return;
    }

    const quarterData = financialsData[requestedYear][quarter];

    if (!quarterData) {
        return;
    }

    return {
        ...quarterData,
        docs: getActualDocs(quarterData.docs, quarterData.date)
    };
}
