import cfg from '@yandex-int/yandex-cfg';
import { BunkerFinancials, BunkerFinancialYear } from 'client/bundles/types';
import { NextFunction, Request, Response } from 'express';
import blogsProvider from 'server/providers/blogs-provider';
import { getReorderedReports, getYearsWithData } from 'server/utils/bunker-utils';
import { getCommonPageData, getFinancialPageUrl } from 'server/utils/page-utils';

const _ = require('lodash');

export default function main(req: Request, res: Response, next: NextFunction) {
    const pageName = 'main';
    const { bunker } = req;

    const financialReports = getLastFinancialReports(req, bunker.pages.financials);

    blogsProvider.getPressReleases(req, { postName: 'posts' })
        .then(pressReleases => {
            res.renderBundle('desktop', pageName, {
                ...getCommonPageData(req, res, pageName),
                main: {
                    ...bunker.pages.main,
                    stocksTeaserUrl: cfg.stocksTeaserUrl,
                    pressReleases: pressReleases.splice(0, 3),
                    lastFinancialReports: getReorderedReports(financialReports, 2)
                }
            });
        })
        .catch(() => next());

}

function getLastFinancialReports(req: Request, financialsData: BunkerFinancials | undefined): BunkerFinancialYear {
    if (!financialsData) {
        return {};
    }

    const reports: BunkerFinancials = _.cloneDeep(financialsData);

    delete reports.title;

    const years = getYearsWithData(financialsData);

    return years
        // Нужно получить только самые последние отчёты.
        // Для этого достаточно парсить только два последение года.
        .splice(0, 2)
        // Начинаем подклеивать с предыдущего года, чтобы отчеты текущего подклеивались сверху
        .reverse()
        .reduce((lastReports, year) => {
            const currentYearReports = reports[year];
            const reportsYear = Object
                .keys(currentYearReports)
                .reduce((sanitizedYearReports, name) => {
                    sanitizedYearReports[name] = {
                        ...currentYearReports[name],
                        html: '',
                        url: getFinancialPageUrl(req, { year, name })
                    };

                    return sanitizedYearReports;
                }, {} as BunkerFinancialYear);

            lastReports = {
                ...lastReports,
                ...reportsYear
            };

            return lastReports;
        }, {} as BunkerFinancialYear);
}
