import url from 'url';

import xml from 'xml';

import LRU from 'lru-cache';

// @ts-ignore
import strftime from 'fast-strftime';

import { NextFunction, Request, Response } from 'express';

import cfg from '@yandex-int/yandex-cfg';
import { PressReleaseItemData } from 'client/bundles/types';
import blogsProvider from 'server/providers/blogs-provider';

const _ = require('lodash');

const cache = new LRU<string, Promise<string | undefined>>({ maxAge: cfg.blogsApi.updateIntervalRss });

const POSTS_COUNT = 20;

export default (req: Request, res: Response, next: NextFunction) => {
    const cacheName = 'press-releases-rss';

    if (!cache.has(cacheName)) {
        cache.set(cacheName, getRssFeed(req));
    }

    // @ts-ignore
    cache
        .get(cacheName)
        .then(feed => {
            res.header('Content-Type', 'text/xml; charset=UTF-8');
            res.send(feed);
        })
        .catch(error => {
            console.error(error);
            cache.del(cacheName);
            next();
        });
};

function getRssFeed(req: Request): Promise<string | undefined> {
    return blogsProvider.getPressReleases(req, { postName: 'posts', entriesNumber: POSTS_COUNT })
        .then(pressReleases => {
            const data = {
                title: 'Yandex Press Releases',
                link: 'https://ir.yandex/press-releases/',
                description: 'Yandex latest news',
                more: 'The full version of the press release is available at:',
                items: pressReleases.length > 0 ? createFeed(req, getNotIgnoredPressReleases(pressReleases)) : []
            };

            // tslint:disable-next-line:no-any
            type Channel = Record<string, any>;

            let channel: Channel[] = [
                { title: data.title },
                { link: data.link },
                { description: data.description }
            ];

            const channelItems = data.items.map((item: PressReleaseItemData) => {
                if (!item.title || !item.url) {
                    return;
                }

                const annotation = `${item.annotation}<p>${data.more} <a href="${item.url}">${item.url}</a></p>`;

                return {
                    item: [
                        { title: item.title },
                        { pubDate: item.date },
                        { description: { _cdata: annotation } },
                        { guid: item.url },
                        { link: item.url }
                    ]
                };
            }).filter(Boolean);

            if (channelItems.length > 0) {
                channel = channel.concat(channelItems);
            }

            return xml({
                rss: [
                    {
                        _attr: {
                            version: '2.0'
                        }
                    },
                    { channel }
                ]
            });
        })
        .catch(() => undefined);
}

function getNotIgnoredPressReleases(items: PressReleaseItemData[]) {
    if (items.length === 0) {
        return [];
    }

    return items.filter((item: PressReleaseItemData) => !item.ignoreInIndex);
}

function createFeed(req: Request, feeds: PressReleaseItemData[]): PressReleaseItemData[] {
    const result = [...feeds];

    const prevTime = new Date(result[result.length - 1].date);
    let count = 1;

    _.forEachRight(result, (item: PressReleaseItemData) => {
        if (!item.url) {
            return;
        }

        const parsedUrl = url.parse(item.url);
        const iDate = item.date;
        let time = new Date(iDate).getTime();

        if (!parsedUrl.host) {
            item.url = url.format({
                protocol: req.protocol,
                host: req.headers.host,
                pathname: parsedUrl.pathname,
                search: parsedUrl.search
            });
        }

        if (time === prevTime.getTime()) {
            // Время новостей с одной датой должно отличаться на минуту
            time += count * 60000;
            count += 1;
        }

        item.date = strftime.strftimeTZ('%a, %d %b %Y %T %z', new Date(time), '+0300');
    });

    return result;
}
