import { FilingType, SecFiling, SecFilingType } from 'client/bundles/types';
import { Request, Response } from 'express';
import moment from 'moment';
import { getCurrentYear } from 'server/utils/bunker-utils';
import { getCommonPageData } from 'server/utils/page-utils';

interface Filters {
    year: string;
    group?: string[];
}

export default function sec(req: Request, res: Response) {
    const pageName = 'sec';
    const { secFilings } = req;

    const years = getYears(secFilings);
    const currentYear = getCurrentYear(req, years);
    const groupId = getCurrentGroupId(req);
    const currentGroup = getCurrentGroup(req, groupId);

    const commonPageData = getCommonPageData(req, res, pageName);

    res.renderBundle('desktop', pageName, {
        ...commonPageData,
        sec: {
            groupId,
            currentYear,
            years,
            types: getTypes(req),
            yearFilings: getFilteredYearFilings(secFilings, { year: currentYear, group: currentGroup }),

            // Запросили конкретный отчёт.
            singleFiling: secFilings && req.query.id && typeof req.query.id === 'string' ?
                getSingleFiling(req, secFilings) :
                undefined
        }
    });
}

function getGroups(req: Request): SecFilingType[] {
    const secGroups = req.bunker.libs?.sec_groups?.items;

    if (!secGroups || secGroups.length === 0) {
        return [];
    }

    return secGroups;
}

function getTypes(req: Request): FilingType[] {
    return getGroups(req).map((group: SecFilingType) => {
        return {
            id: group.groupID,
            value: group.groupName
        };
    });
}

function getCurrentGroup(req: Request, id: string): string[] | undefined {
    const currentGroup = getGroups(req).find(group => group.groupID === id);

    return currentGroup?.groupFilings;
}

function getCurrentGroupId(req: Request): string {
    const requestedType = req.query.type ? (req.query.type as string).trim() : undefined;

    const isExist = getTypes(req).find(type => type.id === requestedType);

    return requestedType && isExist ? requestedType : 'all';
}

function getFilteredYearFilings(secFilings: SecFiling[] | undefined, params: Filters): SecFiling[] {
    if (!secFilings || secFilings.length === 0) {
        return [];
    }

    const date = moment(params.year);

    let result = secFilings.filter(secFiling => date.isSame(secFiling.pubDate, 'year'));

    if (params.group && params.group.length > 0) {
        result = result.filter(secFiling => params.group!.includes(secFiling.formType));
    }

    return result;
}

function getYears(secFilings?: SecFiling[]): string[] {
    if (!secFilings || secFilings.length === 0) {
        return [];
    }

    const set = new Set();

    secFilings.forEach(secFiling => {
        const year = moment(secFiling.pubDate).year();

        set.add(String(year));
    });

    return [...set].sort().reverse() as string[];
}

function getSingleFiling(req: Request, secFilings: SecFiling[]): SecFiling | undefined {
    const { id } = req.query;
    const singleFiling = secFilings.find(secFiling => secFiling.guid === id);

    if (!singleFiling) {
        return;
    }

    return {
        ...singleFiling,
        filingGroup: getGroups(req)
            .filter(group => group.groupFilings.includes(singleFiling.formType))
            .map(group => group.groupName)
            .join(', ')
    };
}
