import { Curl } from 'node-libcurl';

import cfg from '@yandex-int/yandex-cfg';

interface CurlResponse {
    statusCode: number;
    body?: Buffer;
    error?: string;
    headers: Record<string, string>;
}

// Curl.feature.NO_DATA_PARSING = true;

export class TimeoutError extends Error {}

// tslint:disable-next-line:no-any
const normalizeHeaders = (headersRow: any): Record<string, string> => {
    const keysToLowerCase = (obj: Record<string, string>) => {
        return Object.keys(obj).reduce((res, key) => {
            res[key.toString().toLowerCase()] = obj[key];

            return res;
        }, {} as Record<string, string>);
    };

    if (!Array.isArray(headersRow)) {
        return keysToLowerCase(headersRow);
    }

    return headersRow.reduce((res, el) => ({
        ...res,
        ...keysToLowerCase(el)
    }), {});
};

interface CurlRequestOptions {
    headers: Record<string, string>;
    proxyHost: string;
    proxyPort: number;
    timeout?: number;
}

export const curlRequest = (
    url: string,
    options: CurlRequestOptions
): Promise<CurlResponse> => {
    const curl = new Curl();
    const { headers, proxyHost, proxyPort, timeout = 2000 } = options;

    curl.enable(1);
    curl.setOpt(Curl.option.URL, url);
    curl.setOpt(Curl.option.HTTPHEADER, Object.keys(headers).map(key => `${key}: ${headers[key]}`));
    curl.setOpt(Curl.option.CUSTOMREQUEST, 'GET');

    curl.setOpt(Curl.option.PROXY, proxyHost);
    curl.setOpt(Curl.option.PROXYPORT, proxyPort);

    if (['local', 'development'].indexOf(cfg.environment) !== -1) {
        curl.setOpt(Curl.option.VERBOSE, true);
    }

    return new Promise(resolve => {
        try {
            // eslint-disable-next-line no-shadow
            curl.on('end', (statusCode: number, body: Buffer, headers: Record<string, string>) => {
                headers = normalizeHeaders(headers);

                // Несколько раз получала тут 504 ошибку
                if (statusCode !== 200) {
                    console.log(`[ERROR] Ошибка при получении данных через ${proxyHost}:${proxyPort}; statusCode:${statusCode}`);
                }

                curl.close();
                resolve({ statusCode, body, headers });
            });

            // tslint:disable-next-line:no-any
            curl.on('error', (...args: any) => {
                curl.close();

                resolve({
                    statusCode: 500,
                    error: `[ERROR] Ошибка при нормализации данных через ${proxyHost}:${proxyPort}\n${args}`,
                    headers
                });
            });

            curl.perform();
        } catch (err) {
            if (curl.isRunning) {
                curl.close();
            }
            resolve({
                statusCode: 500,
                error: `[ERROR] Ошибка при получении данных через ${proxyHost}:${proxyPort}\n${err}`,
                headers
            });
        }

        setTimeout(() => {
            if (curl.isRunning) {
                curl.close();
                resolve({
                    statusCode: 500,
                    error: `[ERROR] Превышено время ожидания ответа от ${proxyHost}:${proxyPort} больше ${timeout}ms`,
                    headers
                });
            }
        }, timeout);
    });
};
