import cfg from '@yandex-int/yandex-cfg';
import htmlescape from 'htmlescape';
import React from 'react';
import { renderToStaticMarkup, renderToString } from 'react-dom/server';
import Helmet, { HelmetData } from 'react-helmet';
import { StaticRouterContext } from 'react-router';
import { StaticRouter } from 'react-router-dom';

import { BundleData, PageName } from 'client/bundles/types';
import vendorsMeta from 'webpack-configs/vendors-meta';

function getBundle(bundleName: string, lang: string) {
    const module = `../../../ssr.bundles.${lang}`;

    if (cfg.render && cfg.render.isHot) {
        delete require.cache[require.resolve(module)];
    }

    return require(module).bundles[bundleName]; // eslint-disable-line global-require
}

interface RenderBundleArguments {
    bundleName: string;
    data: BundleData;
    lang: string;
    location: string;
    pageName: PageName;
}

/* eslint-disable-next-line max-statements, complexity */
export default ({ bundleName, data, lang, location, pageName }: RenderBundleArguments) => {
    const context: StaticRouterContext = {};
    const Bundle = getBundle(bundleName, lang);

    data.yandexServiceId = cfg.yandexServiceId;
    data.pageName = pageName;

    if (!Bundle) {
        throw new Error(`Bundle ${bundleName} not found`);
    }

    const bundleHtml = renderToString(
        <StaticRouter context={context} location={location}>
            <Bundle data={data} />
        </StaticRouter>
    );

    if (context.url) {
        return { redirectUrl: context.url };
    }

    const helmet = Helmet.rewind();

    return {
        html: getPageHtml({
            helmet,
            bundleName,
            bundleHtml,
            data,
            lang,
            pageName
        })
    };
};

interface PageHtmlParams {
    helmet: HelmetData;
    bundleName: string;
    bundleHtml: string;
    data: BundleData;
    lang: string;
    pageName: string;
}

function getPageHtml(params: PageHtmlParams) {
    const { helmet, bundleName, bundleHtml, data, lang, pageName } = params;
    const { baseUrl, version, frozenPath } = cfg.static;
    const bundleFilePath = `${baseUrl}${version}/${bundleName}.bundle`;
    const vendorsFilePath = `${baseUrl}${frozenPath}/${vendorsMeta.name}`;

    const isNewsPage = pageName === 'pressReleases';
    const isNeedEasyXDM = ['stocks', 'emailAlerts'].includes(pageName);

    const html = renderToStaticMarkup(
        <html lang={`${lang}`}>
            <head>
                {helmet.title.toComponent()}
                {helmet.meta.toComponent()}
                {helmet.link.toComponent()}
                {helmet.script.toComponent()}
                <link rel="stylesheet" href={`${bundleFilePath}.css`} />
                {vendorsMeta.hasCss && <link rel="stylesheet" href={`${vendorsFilePath}.css`} />}
                {isNeedEasyXDM && <script nonce={data.nonce} src={cfg.easyXDM} />}
                {isNewsPage && <link rel="alternate" type="application/rss+xml" title="Press Releases" href={`/press-releases/news.rss`} />}
            </head>
            <body>
                <div id="mount" dangerouslySetInnerHTML={{ __html: bundleHtml }} />
                {vendorsMeta.hasJs && <script nonce={data.nonce} src={`${vendorsFilePath}.js`} />}
                <script nonce={data.nonce} src={`${bundleFilePath}.${lang}.js`} />
                <script
                    nonce={data.nonce}
                    dangerouslySetInnerHTML={{
                        __html: `Client.default(${htmlescape(data)})`
                    }}
                    />
            </body>
        </html>
    );

    return `<!DOCTYPE html>${html}`;
}
