import fs from 'fs';
import path from 'path';

import cfg from '@yandex-int/yandex-cfg';
import { SecFiling } from 'client/bundles/types';
import { NextFunction, Request, Response } from 'express';
import LRU from 'lru-cache';
import getSecFilings from 'server/providers/sec-filings-provider';

const cache = new LRU<string, Promise<SecFiling[] | undefined>>({ maxAge: cfg.secFilings.updateInterval });
let fallbackFileContent: SecFiling[] = [];

const fallbackDirPath = path.join(__dirname, '../../fallback');
const fallbackFilePath = path.join(fallbackDirPath, 'sec-filings.json');

export default (req: Request, _res: Response, next: NextFunction) => {
    const cacheName = 'sec-fillings';

    if (!cache.has(cacheName)) {
        const secFilings = getSecFilings(req);

        cache.set(cacheName, secFilings);
        saveFileSecFilings(secFilings);
    }

    // @ts-ignore
    cache
        .get(cacheName)
        .then((secFilings: SecFiling[]) => {
            if (secFilings) {
                console.log('[SecFilings] Filings count = ', secFilings.length);

                req.secFilings = secFilings;
            } else {
                // Сбросить кеш, чтобы следующий запрос опять постучался в ZORA
                cache.del(cacheName);

                if (fs.existsSync(fallbackFilePath)) {
                    // eslint-disable-next-line max-depth
                    try {
                        console.log('[SecFilings] Парсинг SecFilings из файлового кеша');

                        fallbackFileContent = JSON.parse(fs.readFileSync(fallbackFilePath, 'utf8'));
                    } catch (e) {
                        console.log('[WARNING] Не удалось распарсить SecFilings из файлового кеша', e);
                    }
                } else {
                    console.log('[WARNING] Не обнаружено файлового кеша SecFilings');
                }

                // Отобразить последние закешированные данные
                req.secFilings = fallbackFileContent;
            }
            next();
        })
        .catch((error: Error) => {
            console.error(error);
            cache.del(cacheName);
            next();
        });

    function saveFileSecFilings(promise: Promise<SecFiling[] | undefined>): void {
        promise
            .then((secFilings: SecFiling[]) => {
                if (!secFilings) {
                    return;
                }

                if (!fs.existsSync(fallbackDirPath)) {
                    try {
                        fs.mkdirSync(fallbackDirPath);
                    } catch (e) {
                        console.log('[WARNING] Не удалось создать директорию для файлового кеша SecFilings', e);
                    }
                }

                try {
                    console.log('[SecFilings] Сохранение SecFilings в файл');

                    // Сохранить в памяти свежие данные на случай падения EQS
                    fs.writeFileSync(fallbackFilePath, JSON.stringify(secFilings), {
                        encoding: 'utf8',
                        flag: 'w+',
                        mode: 0o777
                    });
                } catch (e) {
                    console.log('[WARNING] Не удалось сохранить файловый кеш SecFilings', e);
                }
            });
    }

};
