import { format } from 'url';

import cfg from '@yandex-int/yandex-cfg';

import { Request } from 'express';

import request from 'server/lib/request';

import { PostImage, PostTag, PressReleaseItemData } from 'client/bundles/types';

const _ = require('lodash');

interface BlogOptions {
    url: string;
    query: {
        lang: string;
        year?: string;
        size?: number;
    };
    timeout: number;
    headers?: {
        [key: string]: string;
    };
}

interface Post {
    _id: string;
    publishDate: string;
    authorId: string;
    slug: string;
    metaDescription: string;
    metaKeywords: string;
    viewType: string;
    commentsCount: number;
    showPreview: boolean;
    tags: PostTag[];
    ignoreInIndex: boolean;
    ignoreByWidget: boolean;
    financialResults: boolean;
    sourceLink?: string;
    editing: {
        isBlocked: boolean;
    };
    isDraft: boolean;
    approvedTitle: string;
    approvedBody: {
        html: string;
    };
    approvedPreview: {
        html: string;
    };
    hasNext: boolean;
    isMigrated?: boolean;
    socialImage?: PostImage;
}

interface BlogProviderParams {
    postName: string;
    year?: string;
    postId?: string;
    entriesNumber?: number;
}

const blogsProvider = {
    getPressReleases(req: Request, params: BlogProviderParams): Promise<PressReleaseItemData[]> {
        return getPosts(req, params).then(posts => {
            if (!posts.length) {
                return [];
            }

            const prPosts = posts.map(post => buildPressRelease(post));

            return _.sortBy(prPosts, ['date', 'name']).reverse();
        });
    },

    getPressRelease(req: Request, params: BlogProviderParams): Promise<PressReleaseItemData | {}> {
        return getPost(req, params)
            .then(post => {
                if (_.isEmpty(post)) {
                    return {};
                }

                return buildPressRelease(post as Post);
            });
    },

    subscribe(req: Request, email: string): Promise<string> {
        const { protocol, host, blogId, path } = cfg.blogsApi;

        const url = format({
            protocol,
            host,
            pathname: path.subscribe.replace('{{blogId}}', blogId)
        });

        const options = {
            method: 'POST',
            json: true,
            body: { email },
            headers: {
                'x-ya-service-ticket': req?.tvm?.[cfg.tvm.clientId].tickets?.blogsApi.ticket,
                'force-tvm-check': '1'
            }
        };

        return request(url, options);
    }
};

// Возвращает список постов
function getPosts(req: Request, params: BlogProviderParams): Promise<Post[]> {
    const { url, ...options } = getBlogsOptions(req, params);

    return request(url, options)
        .then((posts: string) => {
            if (!posts) {
                throw new Error('Blog posts are undefined.');
            }

            if (typeof posts === 'string') {
                try {
                    posts = JSON.parse(posts);
                } catch (e) {
                    throw new Error('Blog posts type data is invalid');
                }
            }

            return posts as unknown as Post[];
        })
        .catch(error => {
            console.log('Get posts error = ', error.message);

            return [];
        });
}

// Возвращает один пост
function getPost(req: Request, params: BlogProviderParams): Promise<Post | {}> {
    const { url, ...options } = getBlogsOptions(req, params);

    return request(url, options)
        .then((post: string) => {
            if (!post) {
                throw new Error('Blog post is undefined.');
            }

            if (typeof post === 'string') {
                try {
                    post = JSON.parse(post);
                } catch (e) {
                    throw new Error('Blog posts is not an object.');
                }
            }

            return post;
        })
        .catch(error => {
            console.log('Get post error = ', error.message);

            return {};
        });
}

// Возвращает пост в формате пресс-релиза.
function buildPressRelease(post: Post): PressReleaseItemData {
    let url;

    // Некоторые посты не являются пресс-релизами. Например, финасовые релизы.
    // Они приходят с ссылкой на другую страницу.
    if (post.sourceLink) {
        url = post.sourceLink;
        const parsedUrl = new URL(post.sourceLink);

        // В урлах зашит продакшн-хост. Отрезаем, чтобы ссылки всегда указывали на текущее окружение.
        if (parsedUrl.hostname?.includes('ir.yandex')) {
            url = format({ pathname: parsedUrl.pathname, search: parsedUrl.search });
        }
    } else {
        const year = new Date(post.publishDate).getFullYear();
        let migratedSlug;

        if (post.isMigrated) {
            // 2018--slash--0506--dash--1 ---> 0506_1
            migratedSlug = post.slug.replace('--dash--', '_')
                .split('--slash--')
                .pop();
        }
        url = `/press-releases?year=${year}&id=${migratedSlug || post.slug}`;
    }

    return {
        ...buildPost(post),
        url,
        tagsArray: post.tags,
        description: post.metaDescription,
        keywords: post.metaKeywords,

        ignoreInIndex: post.ignoreInIndex,
        ignoreByWidget: post.ignoreByWidget,
        financialResults: post.financialResults,
        isMigrated: post.isMigrated
    } as PressReleaseItemData;
}

// Трансформирует данные поста из blogApi-формата в более читаемый.
function buildPost(post: Post): Partial<PressReleaseItemData> {
    return {
        title: post.approvedTitle,
        date: post.publishDate,
        markup: post.approvedBody && post.approvedBody.html,
        annotation: post.approvedPreview && post.approvedPreview.html,
        socialImage: post.socialImage && post.socialImage.orig && post.socialImage.orig.fullPath
    };
}

// Возвращает параметры запроса для похода в АПИ Блогов.
function getBlogsOptions(_req: Request, params: BlogProviderParams): BlogOptions {
    const { protocol, port, host, path } = cfg.blogsApi;

    let pathname = path[params.postName];

    if (params.year && params.postId) {
        if (isPostFromBunker(params.year, params.postId)) {
            pathname = pathname.replace('{{postId}}', `${params.year}--slash--${params.postId.replace('_', '--dash--')}`);
        } else {
            pathname = pathname.replace('{{postId}}', params.postId);
        }
    }

    const options: BlogOptions = {
        url: format({ protocol, port, host, pathname }),
        query: {
            lang: 'en-EN',
            year: params.year,
            size: params.entriesNumber
        },
        timeout: 3000
    };

    return options;
}

const MIGRATION_YEAR = 2018;
const MIGRATION_MONTH = 5;
const MIGRATION_DAY = 30;

function isPostFromBunker(year: string, postId: string): boolean {
    const postYear = Number(year);

    // Все посты раньше 2018 года — из Бункера, а не из Блогов.
    if (postYear < MIGRATION_YEAR) {
        return true;
    }

    // Если не удовлетворяют 0224, 0225_1 — точно не из Бункера.
    if (!postId.match(/^\d{4}(_\d)?$/)) {
        return false;
    }

    postId = postId.replace(/_\d/, '');
    const postMonth = parseInt(postId.substr(0, 2), 10);
    const postDate = postId.substr(2);

    const postFullDate = new Date(postYear, postMonth - 1, Number(postDate));
    const migrationDate = new Date(MIGRATION_YEAR, MIGRATION_MONTH - 1, MIGRATION_DAY);

    return postFullDate.getTime() <= migrationDate.getTime();
}

export default blogsProvider;
