import cfg from '@yandex-int/yandex-cfg';
import { SecFiling } from 'client/bundles/types';
import { Request } from 'express';
import { curlRequest } from 'server/lib/curl';

async function getSecFilings(req: Request): Promise<SecFiling[] | undefined> {
    const tvmTicket = req?.tvm?.[cfg.tvm.clientId].tickets?.zora.ticket;

    if (!tvmTicket) {
        console.log('[ERROR] Отсутствует tvmTicket');

        return undefined;
    }

    const { timeout, sourceName, host, port, cacheTime } = cfg.zora;

    // @see: https://wiki.yandex-team.ru/zora/userguide/#specialnyezagolovkivhttpzaprosekzora
    const headersZora: Record<string, string> = {
        'X-Ya-Service-Ticket': tvmTicket,
        'X-Yandex-Sourcename': sourceName,
        'X-Yandex-Requesttype': 'online',
        'X-Yandex-Use-Https': '1',
        'Cache-Control': `max-age=${cacheTime / 1000}`, // Zora demands cache time timeout in seconds
        'X-Yandex-Response-Timeout': String(timeout / 1000), // Zora demands response timeout in seconds
        'X-Yandex-Redirs': '1'
    };

    const { body, error, headers } = await curlRequest(cfg.secFilings.url, { headers: headersZora, proxyHost: host, proxyPort: port, timeout });

    if (!body || error) {
        console.log(`[ERROR] Ошибка при запросе sec-filings\n${error || ''}`);

        return undefined;
    }

    if (headers.date) {
        console.log(`[SecFilings] Дата получения данных от ${host}: ${headers.date}`);
    }

    try {
        const rawSecFilings = body.toString();

        return parseSecFilings(JSON.parse(rawSecFilings));
    } catch (e) {
        console.log(`[ERROR] Ошибка при парсинге sec-filings\n${e}`);

        return undefined;
    }
}

function parseSecFilings(secFilings: SecFiling[]): SecFiling[] {
    return secFilings.map(secFiling => {
        return {
            ...secFiling,
            documentFormatFiles: secFiling.documentFormatFiles.map(file => {
                return {
                    ...file,
                    link: file.link && file.link.replace(/&amp;/gmi, '&')
                };
            })
        };
    });
}

export default getSecFilings;
