import { ErrorRequestHandler, RequestHandler, Router } from 'express';

import analysts from 'server/controllers/analysts';
import bonds from 'server/controllers/bonds';
import bondsDocument from 'server/controllers/bonds-document';
import contacts from 'server/controllers/contacts';
import contactsGovernance from 'server/controllers/contacts-governance';
import directors from 'server/controllers/directors';
import emailAlerts from 'server/controllers/email-alerts';
import events from 'server/controllers/events';
import eventsRemind from 'server/controllers/events-remind';
import faq from 'server/controllers/faq';
import faqGovernance from 'server/controllers/faq-governance';
import financials from 'server/controllers/financials';
import keyBusinesses from 'server/controllers/key-businesses';
import keyDocuments from 'server/controllers/key-documents';
import latestPresentation from 'server/controllers/latest-presentation';
import main from 'server/controllers/main';
import mainDocument from 'server/controllers/main-document';
import management from 'server/controllers/management';
import meetings from 'server/controllers/meetings';
import notFound from 'server/controllers/not-found';
import oldFinancial from 'server/controllers/old-financial';
import ourPrinciples from 'server/controllers/our-principles';
import overview from 'server/controllers/overview';
import pressReleases from 'server/controllers/press-releases';
import pressReleasesRss from 'server/controllers/press-releases-rss';
import robots from 'server/controllers/robots';
import sec from 'server/controllers/sec';
import stocks from 'server/controllers/stocks';
import structure from 'server/controllers/structure';
import subscribe from 'server/controllers/subscribe';
import sustainability from 'server/controllers/sustainability';
import sustainabilityDocument from 'server/controllers/sustainability-document';
import sustainabilityWebcast from 'server/controllers/sustainability-webcast';
import secFilings from 'server/middlewares/sec-filings';

import {
    bunker,
    cookieParser,
    helmet,
    httpUatraits,
    tld,
    tvm,
    xForwardedForFix,
    yandexCsp
} from 'server/middlewares';

const middlewares: Array<RequestHandler | ErrorRequestHandler> = [
    bunker,
    cookieParser,
    tld,
    tvm,
    helmet,
    yandexCsp, // Depends on cookieParser
    xForwardedForFix,
    httpUatraits
];

// eslint-disable-next-line max-statements
export default function (router: Router) {
    router.get('/', middlewares, main);
    router.get('/robots.txt', robots);
    router.get('/subscribe/', tvm, subscribe);

    router.get('/financial-releases', middlewares, financials);
    router.get('/events-and-presentations', middlewares, events);
    router.get('/events-and-presentations/:fileName', middlewares, eventsRemind);
    router.get('/sec-filings', middlewares, secFilings, sec);
    router.get('/analyst-coverage', middlewares, analysts);
    router.get('/faq', middlewares, faq);
    router.get('/email-alerts', middlewares, emailAlerts);
    router.get('/contact-us', middlewares, contacts);
    router.get('/press-releases/news.rss', middlewares, pressReleasesRss);
    router.get('/press-releases', middlewares, pressReleases);
    router.get('/our-principles', middlewares, ourPrinciples);
    router.get('/key-businesses', middlewares, keyBusinesses);

    router.get('/convertible-bonds', middlewares, bonds);
    router.get('/convertible-bonds/notice-to-noteholders', middlewares, bondsDocument);
    router.get('/convertible-bonds/tise-notes-delisting-notice', middlewares, bondsDocument);
    router.get('/convertible-bonds/delisting-event-notice', middlewares, bondsDocument);

    router.get('/sustainability', middlewares, sustainability);
    router.get('/sustainability/report-en', middlewares, sustainabilityDocument);
    router.get('/sustainability/report-ru', middlewares, sustainabilityDocument);
    router.get('/sustainability/presentation', middlewares, sustainabilityDocument);
    router.get('/sustainability/webcast', middlewares, sustainabilityWebcast);

    router.get('/latest-presentation', middlewares, latestPresentation);

    router.get('/shareholder-structure', middlewares, structure);
    router.get('/shareholder-meetings', middlewares, meetings);
    router.get('/stock-information', middlewares, stocks);

    router.get('/corporate-governance', middlewares, overview);
    router.get('/corporate-governance/overview', middlewares, overview);
    router.get('/corporate-governance/board-of-directors', middlewares, directors);
    router.get('/corporate-governance/key-documents', middlewares, keyDocuments);
    router.get('/corporate-governance/management', middlewares, management);
    router.get('/corporate-governance/faq', middlewares, faqGovernance);
    router.get('/corporate-governance/contacts', middlewares, contactsGovernance);

    router.get('/databook', middlewares, mainDocument);
    router.get('/latest-annual-report', middlewares, mainDocument);

    router.get('/releasedetail.cfm', middlewares, oldFinancial);

    router.use(middlewares, notFound);
}
