import { Request } from 'express';

import {
    BunkerData,
    BunkerEvents,
    BunkerFinancials,
    BunkerFinancialType,
    BunkerFinancialYear,
    BunkerSustainabilityLinkType,
    Doc,
    MenuItem,
    PrefooterData,
    SUSTAINABILITY_TYPES
} from 'client/bundles/types';
import moment from 'moment';

const _ = require('lodash');

const COUNT_YEAR_EXPIRED_WEBCAST = 2;

// eslint-disable-next-line import/prefer-default-export
export function getMenuFromBunker(bunker: BunkerData, pageName?: string): MenuItem[] {
    const { items } = _.cloneDeep(bunker.common.menu);

    if (!pageName) {
        return items;
    }

    return items.map((item: MenuItem) => {
        if (item.id === pageName) {
            item.isCurrent = true;
        }

        return item;
    });
}

export function getPrefooter(bunker: BunkerData): PrefooterData {
    return {
        ...bunker.common.prefooter
    };
}

export function getYearsWithData(dataWithYears?: BunkerEvents| BunkerFinancials): string[] {
    if (!dataWithYears) {
        return [];
    }

    const years = Object.keys(dataWithYears).filter(key => {
        // Оставляем только поля с годом, в которых есть данные.
        if (!/^20\d{2}$/.test(key) || typeof dataWithYears[key] !== 'object') {
            return;
        }

        return Object.values(dataWithYears[key]).filter(Boolean).length > 0;
    });

    return years.sort().reverse();
}

export function getCurrentYear(req: Request, yearList: string[]): string {
    const requestedYear = req.query.year ? (req.query.year as string).trim() : undefined;

    return requestedYear && /^20\d{2}$/.test(requestedYear) && yearList.includes(requestedYear) ?
        requestedYear :
        yearList[0];
}

export function getReorderedReports(reports: BunkerFinancialYear, maxCount?: number): BunkerFinancialYear {
    const reordered = {} as BunkerFinancialYear;

    const sortReportsKeys = Object.keys(reports).sort((a: BunkerFinancialType, b: BunkerFinancialType) => {
        return new Date(reports[b].date).getTime() - new Date(reports[a].date).getTime();
    });

    sortReportsKeys.forEach((key: BunkerFinancialType, index) => {
        if (!maxCount || index < maxCount) {
            reordered[key] = reports[key];
        }
    });

    return reordered;
}

function isValidSustainabilityType(name?: string): name is BunkerSustainabilityLinkType {
    return (SUSTAINABILITY_TYPES as readonly string[]).includes(name || '');
}

export function getSustainabilityType(name?: string): BunkerSustainabilityLinkType | undefined {
    return isValidSustainabilityType(name) ? name : undefined;
}

export function getActualDocs(docs: Doc[], date: string): Doc[] {
    const nowDate = moment();
    const webcastDate = moment(date);

    const dateDiff = nowDate.diff(webcastDate, 'years', true);

    const isExpiredWebcast = dateDiff > COUNT_YEAR_EXPIRED_WEBCAST;

    return docs.filter(doc => {
        return doc.format !== 'webcast' || !isExpiredWebcast;
    });
}
