import csv
import datetime
import json

import requests
import yt
import yt.yson as yson
from requests_ntlm import HttpNtlmAuth

from source.config import *
from source.monitoring.monitoring import SimpleYTBasedCollector
from source.utils import exc_thread_wrapper, OtherUtils

logger = logging.getLogger('CMDB')
yt.config["proxy"]["url"] = "vanga"

YT_CLIENT = make_hahn_client()

class NanimatorYTCollector(SimpleYTBasedCollector):
    def __init__(self):
        super().__init__(yt_table_name='cmdb/nanimator', recreate=False, client=YT_CLIENT)
        self.schema = [
            {'name': 'login', 'type': 'string',
             'sort_order': 'ascending'},
            {'name': 'os', 'type': 'string'},
            {'name': 'position', 'type': 'string'},
            {'name': 'need_phone', 'type': 'boolean'},
            {'name': 'hardware_profile', 'type': 'int64'},
            {'name': 'hdrfs_ticket', 'type': 'string'},
            {'name': 'hardware_profile__profile_id', 'type': 'string'},
            {'name': 'form_type', 'type': 'string'},
            {'name': 'hardware_profile_description', 'type': 'string'},
            {'name': 'need_tv', 'type': 'boolean'},
            {'name': 'candidate_type', 'type': 'string'},
            {'name': 'join_at', 'type': 'string'},
            {'name': 'status', 'type': 'string'},
            {'name': 'office', 'type': 'int64'},
            {'name': 'department', 'type': 'int64'},
            {'name': 'table', 'type': 'int64'},
            {'name': 'room', 'type': 'int64'},
        ]

    def _fetch_data(self):
        result = []
        results = requests.get(
            'https://staff.yandex-team.ru/preprofile-api/export/helpdesk/',
            headers=AUTH_HEADERS_TOOLS).json()
        approved_keys = [x['name'] for x in self.schema]

        for record in results:
            result.append(
                {key: record[key] for key in record if key in approved_keys})

        return result

    def main(self):
        data = self._fetch_data()
        self.check_or_create_table()
        self.write_data_to_yt(data)


class CertificatorYTCollector(SimpleYTBasedCollector):
    def __init__(self):
        super().__init__(yt_table_name='cmdb/staff_certificator',
                         recreate=False,
                         client=YT_CLIENT)
        self.schema = [
            {'name': 'username', 'type': 'string',
             'sort_order': 'ascending'},
            {'name': 'instance_number', 'type': 'string'},
            {'name': 'type', 'type': 'string'},
        ]

    @staticmethod
    def _get_all_certs_for_user(login):
        result = []
        url = 'https://crt-api.yandex-team.ru/api/v2/certificates/?user={}&type=pc&type=linux-pc&_fields=type,issued,pc_inum,pc_serial_number,pc_mac'.format(
            login)

        while url:
            data = requests.get(url, headers=AUTH_HEADERS_CRT).json()
            result.extend(data.get('results'))
            url = data.get('next')

        issued_certs = [x for x in result if x['issued']]
        issued_certs.sort(key=lambda x: x['issued'], reverse=True)

        return issued_certs

    @staticmethod
    def _find_inv_by_sn(sn):
        for item in ['{}', 's{}']:
            search = list(YT_CLIENT.select_rows(
                'instance_number from [{}] where lower(SN) = "{}"'.format(
                    '//home/helpdesk/cmdb/notebooks',
                    item.format(sn.lower())),
                format=yt.JsonFormat(attributes={"encode_utf8": False})
            ))
            if search: return search[0].get('instance_number')
        return search

    @staticmethod
    def _get_month():
        month = datetime.datetime.now().month
        return '0{}'.format(month) if int(month) < 10 else month

    def _get_full_data_by_login(self, login, certificates):
        result = {
            "username": login,
            "type": None,
            "instance_number": None
        }
        if not certificates:
            logger.info('There is not certificate - user {}'.format(login))
            return result

        last_certificate = certificates[0]
        result['instance_number'] = last_certificate.get('pc_inum')
        result['type'] = last_certificate.get('type')
        pc_serial_number = last_certificate.get('pc_serial_number')

        if result['instance_number']:
            return result

        elif self._find_inv_by_sn(pc_serial_number):
            result['instance_number'] = self._find_inv_by_sn(pc_serial_number)
            return result

        return result

    def _fetch_data(self):
        upload_data = []
        yt_data = YT_CLIENT.select_rows(
            'login, quit_at from [{table}] where regex_partial_match("2019-{month}-[0-9]*", quit_at) '.format(
                table='//home/helpdesk/cmdb/staff',
                month=self._get_month()),
            format=yt.JsonFormat(
                attributes={"encode_utf8": False}))

        for login in [x['login'] for x in yt_data]:
            certificates = self._get_all_certs_for_user(login)
            upload_data.append(self._get_full_data_by_login(login, certificates))

        return upload_data

    def main(self):
        data = self._fetch_data()
        self.check_or_create_table()
        self.write_data_to_yt(data)

class OebsNotebooksYTCollector(SimpleYTBasedCollector):
    def __init__(self, *args, **kwargs):
        recreate = kwargs.get('recreate', False)
        super().__init__(yt_table_name='cmdb/notebooks', recreate=recreate, client=YT_CLIENT)
        self.schema = [{'name': 'instance_number', 'type': 'string','sort_order': 'ascending'},
                       {'name': 'SN', 'type': 'string'},
                       {'name': 'ext_login', 'type': 'string'},
                       {'name': 'MACs', 'type': 'string'},
                       {'name': 'hw_type', 'type': 'string'},
                       {'name': 'status_name', 'type': 'string'},
                       {'name': 'oebs_login', 'type': 'string'},
                       {'name': 'org_name', 'type': 'string'},
                       {'name': 'loc_segment1', 'type': 'string'},
                       {'name': 'loc_segment2', 'type': 'string'},
                       {'name': 'loc_segment3', 'type': 'string'},
                       {'name': 'loc_segment4', 'type': 'string'},
                       {'name': 'loc_segment5', 'type': 'string'},
                       {'name': 'loc_segment6', 'type': 'string'},
                       {'name': 'loc_segment7', 'type': 'string'},
                       {'name': 'segment1', 'type': 'string'},
                       {'name': 'segment2', 'type': 'string'},
                       {'name': 'creation_date', 'type': 'string'},
                       {'name': 'active_start_date', 'type': 'string'},
                       {'name': 'active_end_date', 'type': 'string'},
                       {'name': 'last_ticket_number', 'type': 'string'},
                       {'name': 'last_assign_date', 'type': 'string'},
                       {'name': 'fqdn', 'type': 'string'},
                       {'name': 'tech_support_date', 'type': 'string'},
                       {'name': 'vendor', 'type': 'string'},
                       {'name': 'model', 'type': 'string'},
                       {'name': 'screensize', 'type': 'string'},
                       {'name': 'resolution', 'type': 'string'},
                       {'name': 'cpu', 'type': 'string'},
                       {'name': 'disk_type', 'type': 'string'},
                       {'name': 'disk', 'type': 'string'},
                       {'name': 'ram', 'type': 'string'},
                       {'name': 'exp_date', 'type': 'string'},
                       {'name': 'eq_status', 'type': 'string'},
                       {'name': 'os', 'type': 'string'}
                       ]

    def _fetch_data(self):
        logger.info('Start Fetchig data from BOT')
        result = []
        results = requests.get(
            'https://bot.yandex-team.ru/api/view.php?name=view_hd_hardware&format=json',
            headers=AUTH_HEADERS_BOT).json()
        approved_keys = [x['name'] for x in self.schema]
        for record in results:
            if record['hw_type'] in ['USR.NOTEBOOKS','USR.DESKTOPS']:
                result.append(
                    {key: record[key] for key in record if key in approved_keys})

        return result

    def _delete_old_records(self, new_data_raw):
        old_data = list(YT_CLIENT.select_rows(
            'instance_number from [{}]'.format(
                '//home/helpdesk/cmdb/notebooks'),
            format=yt.JsonFormat(attributes={"encode_utf8": False})
        ))
        new_data = [x["instance_number"] for x in new_data_raw]

        diff = [x for x in old_data if x["instance_number"] not in new_data]
        logger.info('DIFF is {}'.format(diff))
        YT_VANGA_CLIENT.delete_rows(self.yt_table,
                                    diff,
                                    format=yt.JsonFormat(attributes={"encode_utf8": False})
                                    )

    def _creation_date_diff_to_eq_status(self, diff_in_days):

        diff = diff_in_days / 365

        if diff >= 0 and diff < 2 :
            return 'actual'
        if diff >= 2 and diff <= 5:
            return 'reserved'
        if diff > 5:
            return 'non_actual'

    @staticmethod
    def _analyze_os_data(raw_data):
        result = []
        data = list(raw_data)
        for item in data:
            os = "macOS" if "APPLE" in item["segment2"].upper() else "Windows"
            result.append({
                "instance_number":item["instance_number"],
                "os":os
            })
        return result

    def _analyze_eq_statuses_data(self, raw_data):
        result = []
        data = list(raw_data)
        for item in data:
            raw_creation_date = item["creation_date"]
            creation_date = datetime.datetime.strptime(raw_creation_date,
                                                       '%Y-%m-%d %H:%M:%S')
            exp_date = creation_date + datetime.timedelta(days=365*3)
            diff_time = abs(datetime.datetime.now() - creation_date).days
            eq_status = self._creation_date_diff_to_eq_status(diff_time)
            result.append({
                "instance_number" : item["instance_number"],
                "exp_date" : exp_date.strftime('%Y-%m-%d %H:%M:%S'),
                "eq_status" : eq_status
            })
        return result

    def _populate_os(self):
        logger.info('Start populate OS')
        yt_data = YT_CLIENT.select_rows('instance_number, segment1, segment2 from [{}] where os = NULL'.format(self.yt_table),format=yt.JsonFormat(attributes={"encode_utf8": False}))
        new_data = self._analyze_os_data(yt_data)
        self.write_data_to_yt(new_data)

    def _populate_eq_status(self):
        logger.info('Start populate eq statuses')
        yt_data = YT_CLIENT.select_rows('instance_number, creation_date from [{}]'.format(self.yt_table),format=yt.JsonFormat(attributes={"encode_utf8": False}))
        new_data = self._analyze_eq_statuses_data(yt_data)
        self.write_data_to_yt(new_data)

    def test(self):
        yt_data = YT_CLIENT.select_rows('instance_number, segment1, segment2 from [{}] limit 10'.format(self.yt_table),format=yt.JsonFormat(attributes={"encode_utf8": False}))
        data = list(yt_data)
        print(data)

    def main(self):
        data = self._fetch_data()
        self._delete_old_records(data)
        self.check_or_create_table()
        self.write_data_to_yt(data)
        self._populate_os()
        self._populate_eq_status()



class YTTableSyncer(SimpleYTBasedCollector):
    def __init__(self, *args, **kwargs):
        self.target_client = kwargs.get('target_client') if kwargs.get('target_client') else yt
        self.source_path = kwargs.get('source_path')
        self.source_client = kwargs.get('source_client') if kwargs.get('source_client') else yt
        recreate = kwargs.get('recreate')
        super().__init__(yt_table_name=kwargs.get('target_path'), recreate=recreate, client=self.target_client)
        self.schema = self._get_schema()

    def _get_schema(self):
        raw_schema = self.source_client.get_attribute(self.source_path, "schema")
        data = yson.convert.yson_to_json(raw_schema)["$value"]
        for dict in data:
            for key in dict:
                value = dict[key]
                dict[key] = {'true':True,'false':False}.get(value, value)
        return data

    def _fetch_data(self):
        logger.info("Start fetching data ST")
        yt_data = self.source_client.select_rows(
            '* from [{}]'.format(
                self.source_path),
            format=yt.JsonFormat(attributes={"encode_utf8": False}))

        return list(yt_data)

    def main(self):
        data = self._fetch_data()
        self.check_or_create_table()
        logger.info("Start upload to YT")
        self.write_data_to_yt(data)

class HyperCubeYTCollector(SimpleYTBasedCollector):
    def __init__(self):
        super().__init__(yt_table_name='cmdb/hypercube', recreate=False, client = YT_VANGA_CLIENT)
        self.schema = [{'name': 'instance_number', 'type': 'string',
                        'sort_order': 'ascending'},
                       {'name': 'SN', 'type': 'string'},
                       {'name': 'ext_login', 'type': 'string'},
                       {'name': 'oebs_login', 'type': 'string'},
                       {'name': 'last_assign_date', 'type': 'string'},
                       {'name': 'vendor', 'type': 'string'},
                       {'name': 'model', 'type': 'string'},
                       {'name': 'segment1', 'type': 'string'},
                       {'name': 'segment2', 'type': 'string'}
                       ]

    def _fetch_data(self):
        result = []
        results = requests.get(
            'https://bot.yandex-team.ru/api/view.php?name=view_hd_hardware&format=json',
            headers=AUTH_HEADERS_BOT).json()
        approved_keys = [x['name'] for x in self.schema]
        for record in results:
            if record['hw_type'] in ['USR.MOBILE']:
                result.append(
                    {key: record[key] for key in record if key in approved_keys})

        return result

    def main(self):
        data = self._fetch_data()
        self.check_or_create_table()
        logger.info("Start upload to YT")
        self.write_data_to_yt(data)

class SCCMSoftCollector(SimpleYTBasedCollector):
    def __init__(self, recreate=False):
        super().__init__(yt_table_name='cmdb/sccm_installed_software', recreate=recreate, client = YT_CLIENT)
        self.schema = [
            {'name': 'instance_number', 'type': 'string',
             'sort_order': 'ascending'},
            {'name': 'ms_office', 'type': 'boolean'},
        ]
        self.seen_invs = set()

    def _collect_sn_and_invs_from_yt(self):
        self.sn_inv_dict = {}

        logger.info('Fetch SN/INV data from YT')
        yt_table = '//home/helpdesk/cmdb/notebooks'
        yt_data = YT_CLIENT.select_rows('instance_number, SN, MACs from [{}]'.format(yt_table),format=yt.JsonFormat(attributes={"encode_utf8": False}))
        for item in yt_data:

            sn_list = [item["SN"].upper(), item["SN"][1:].upper()] if item["SN"] else [item["SN"]]
            mac_list = item["MACs"].split(',') if item.get("MACs") else []

            for value in sn_list + mac_list:
                self.sn_inv_dict[value] = item['instance_number']

    def _parse_inv_num(self, parsed_objects):
        for p_object in parsed_objects:
            parsed_inv = self.sn_inv_dict.get(p_object)
            if parsed_inv:
                return parsed_inv
        return False

    def _jss_to_yt_transformer(self, data):
        result = {}
        for key in data:
            innercase_key = key.lower()
            if innercase_key in [x['name'] for x in self.schema]:
                result[innercase_key] = data[key]

            if innercase_key in ['serial_number','macs'] and data[innercase_key]:
                prepared_list = [x.replace(':','').replace(' ','').upper() for x in data[key].split(',')]
                inv_number = self._parse_inv_num(prepared_list)
                if inv_number:
                    result['instance_number'] = inv_number
                    result['ms_office'] = True
                    self.seen_invs.add(inv_number)
                    return result

        if not result.get('instance_number') :
            logger.warning(
                'Cannot find notebook with sn {} in oebs'.format(data))
            return False
        return result

    def _make_unseen_result_as_false(self, data):
        yt_table = '//home/helpdesk/cmdb/sccm_installed_software'
        try:
            yt_data = YT_CLIENT.select_rows('instance_number from [{}]'.format(yt_table),format=yt.JsonFormat(attributes={"encode_utf8": False}))
        except yt.errors.YtHttpResponseError:
            logger.info('Cannot find jss_installed_software seems its first launch')
            return data
        for item in yt_data:
            if item['instance_number'] not in self.seen_invs:
                data.append({'instance_number':item['instance_number'], 'ms_office':False})

    def _fetch_data(self):
        self._collect_sn_and_invs_from_yt()
        result = []
        URL = 'https://iva-cfgp01-db01.ld.yandex.ru/ReportServer/Pages/ReportViewer.aspx?%2FConfigMgr_Y01%2FYandex%2FComputersByCollection&rs:Format=csv'
        data = requests.get(URL, verify=False,
                            auth=HttpNtlmAuth('LD\\robot-help',
                                              ROBOT_HELP_PWD))
        content = data.content.decode('utf-8').split('\r\n')
        reader = csv.DictReader(content, fieldnames=(
        "computer_name", "user_name", "last_logon", "manufacter","model","serial_number","macs"))
        data = json.loads(json.dumps([row for row in reader]))

        for item in data:
            parsed_dict = self._jss_to_yt_transformer(item)
            if parsed_dict: result.append(parsed_dict)

        self._make_unseen_result_as_false(result)
        return result

    def main(self):
        data = self._fetch_data()
        self.check_or_create_table()
        self.write_data_to_yt(data)

class RepairOebsCollector(SimpleYTBasedCollector):
    def __init__(self):
        super().__init__(yt_table_name='cmdb/repair_oebs', recreate=False, client = YT_CLIENT)
        self.schema = [{'name': 'instance_number', 'type': 'string',
                        'sort_order': 'ascending'},
                       {'name': 'notes', 'type': 'string'},
                       {'name': 'note_type', 'type': 'string'},
                       {'name': 'entered_date', 'type': 'string'}
                       ]

    def _fetch_data(self):
        result = []
        results = requests.get(
            'https://bot.yandex-team.ru/api/view.php?name=view_hd_repairs&format=json',
            headers=AUTH_HEADERS_BOT).json()
        approved_keys = [x['name'] for x in self.schema]
        for record in results:
            result.append(
                {key: record[key] for key in record if key in approved_keys})

        return result

    def main(self):
        data = self._fetch_data()
        self.check_or_create_table()
        logger.info("Start upload to YT")
        self.write_data_to_yt(data)


@exc_thread_wrapper
def main_nanimator():
    logger.info('CMDB Nanimator sync started')
    NanimatorYTCollector().main()

@exc_thread_wrapper
def main_certificator():
    logger.info('CMDB Certificator sync started')
    CertificatorYTCollector().main()

@exc_thread_wrapper
def main_cmdb_notebooks():
    logger.info('CMDB notebooks sync started')
    OebsNotebooksYTCollector().main()

@exc_thread_wrapper
def main_sync_st_from_markov():
    YTTableSyncer(target_path='cmdb/st_issues',
                  target_client=YT_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/queue/HDRFS/issues',
                  ).main()

    YTTableSyncer(target_path='cmdb/st_priorities',
                  target_client=YT_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/common/priorities',
                  ).main()

    YTTableSyncer(target_path='cmdb/st_resolutions',
                  target_client=YT_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/common/resolutions',
                  ).main()

    YTTableSyncer(target_path='cmdb/st_schema_version',
                  target_client=YT_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/common/schema_version',
                  ).main()

    YTTableSyncer(target_path='cmdb/st_statuses',
                  target_client=YT_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/common/statuses',
                  ).main()

    YTTableSyncer(target_path='cmdb/st_types',
                  target_client=YT_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/common/types',
                  ).main()

    YTTableSyncer(target_path='cmdb/st_users',
                  target_client=YT_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/common/users',
                  ).main()

    YTTableSyncer(target_path='cmdb/st_versions',
                  target_client=YT_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/queue/HDRFS/versions',
                  ).main()

    YTTableSyncer(target_path='cmdb/st_components',
                  target_client=YT_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/queue/HDRFS/components',
                  ).main()
@exc_thread_wrapper
def main_hypercube_yt():
    logger.info('Start staff to YT sync')
    HyperCubeYTCollector().main()

@exc_thread_wrapper
def main_sccm_soft_yt():
    logger.info('Start staff to YT sync')
    SCCMSoftCollector().main()

@exc_thread_wrapper
def main_repair_collector():
    logger.info('Repair OEbS Collector')
    RepairOebsCollector().main()

def move_table():
    YTTableSyncer(target_path='cmdb/nanimator',
                  target_client=YT_CLIENT,
                  source_path='//home/helpdesk/cmdb/nanimator_backup',
                  source_client=YT_CLIENT).main()
