from source.config import *
from source.monitoring.monitoring import SimpleYTBasedCollector
from source.utils import exc_thread_wrapper, SerenityCollector

logger = logging.getLogger(__name__)


class AppleMobileDevicesCollector(SimpleYTBasedCollector):
    def __init__(self):
        super().__init__(yt_table_name='cmdb/jss_all/mobile_devices_serenity',
                         recreate=False,
                         client=YT_HAHN_CLIENT,
                         is_table_dynamic=False,
                         static_append=False)

        self.schema = [
            {'name': 'id', 'type': 'int64'},
            {'name': 'name', 'type': 'string'},
            {'name': 'udid', 'type': 'string'},
            {'name': 'username', 'type': 'string'},
            {'name': 'serial_number', 'type': 'string'},
            {'name': 'model', 'type': 'string'},
            {'name': 'last_enrollment', 'type': 'string'},
            {'name': 'last_inventory_update', 'type': 'string'},
            {'name': 'ios_version', 'type': 'string'},
            {'name': 'mail_installed', 'type': 'boolean'},
            {'name': 'vpn_installed', 'type': 'boolean'},
            {'name': 'cal_installed', 'type': 'boolean'},
            {'name': 'contacts_installed', 'type': 'boolean'}
        ]
        self.mob_collect = SerenityCollector()
        self._collect_profiles_info()

    def _collect_sn_by_search_id(self, search_id):
        logger.info('Start collection search with {}'.format(search_id))
        raw_data = self.mob_collect.query(
            query='advancedmobiledevicesearches/id/{}'.format(search_id))
        data = raw_data['advanced_mobile_device_search']['mobile_devices']
        return {x['Serial_Number']:True for x in data}

    def _collect_profiles_info(self):
        self.calendar_set =  self._collect_sn_by_search_id(search_id=31)
        self.contacts_set = self._collect_sn_by_search_id(search_id=33)
        self.mail_set = self._collect_sn_by_search_id(search_id=32)
        self.open_vpn_set = self._collect_sn_by_search_id(search_id=34)

    def _analyze_profiles(self, record):
        sn = record["Serial_Number"]
        return {"mail_installed":self.calendar_set.get(sn, False),
                "vpn_installed":self.open_vpn_set.get(sn, False),
                "contacts_installed": self.contacts_set.get(sn, False),
                "cal_installed": self.calendar_set.get(sn, False),}

    def _analyze_one_record(self, record):
        result = self._analyze_profiles(record)
        for item in record:
            lower_item = item.lower()
            if lower_item in [x["name"] for x in self.schema]:
                result[lower_item] = record[item]
        return result

    def _fetch_data(self):
        result = []
        raw_data =  self.mob_collect.query(query='advancedmobiledevicesearches/id/30')
        data = raw_data['advanced_mobile_device_search']['mobile_devices']
        for item in data:
            result.append(self._analyze_one_record(item))
        return result

    def main(self):
        data = self._fetch_data()
        self.write_data_to_yt(data)

@exc_thread_wrapper
def main_jss_mobile_devices():
    AppleMobileDevicesCollector().main()