import yt
import yt.yson as yson

from source.config import *
from source.monitoring.monitoring import SimpleYTBasedCollector
from source.utils import exc_thread_wrapper

logger = logging.getLogger(__name__)

class STStaticSyncer(SimpleYTBasedCollector):
    def __init__(self, *args, **kwargs):
        self.target_client = kwargs.get('target_client', yt)
        self.source_path = kwargs.get('source_path')
        self.source_client = kwargs.get('source_client', yt)
        super().__init__(yt_table_name=kwargs.get('target_path'),
                         recreate=kwargs.get('recreate'),
                         client=self.target_client,
                         is_table_dynamic=kwargs.get('is_table_dynamic'),
                         static_append=kwargs.get('static_append')
                         )
        self.schema = [
            {
                "name": "key",
                "required": True,
                "type": "string",
            },
            {
                "name": "resolved",
                "required": False,
                "type": "int64"
            },
            {
                "name": "created",
                "required": False,
                "type": "int64"
            },
            {
                "name": "type",
                "required": True,
                "type": "string",
            },
            {
                "name": "fixVersions",
                "required": False,
                "type": "int64"
            },
            {
                "name": "abcservice",
                "required": False,
                "type": "int64"
            },
            {
                "name": "sla_604_spent",
                "required": False,
                "type": "int64"
            },
            {
                "name": "sla_604_status",
                "required": False,
                "type": "string"
            },
            {
                "name": "sla_605_spent",
                "required": False,
                "type": "int64"
            },
            {
                "name": "sla_605_status",
                "required": False,
                "type": "string"
            },
            {
                "name": "resolution",
                "required": False,
                "type": "string"
            }
        ]
        self.dict_ver = self._fetch_data_st_fixvers()

    def _get_schema(self):
        raw_schema = self.source_client.get_attribute(self.source_path, "schema")
        data = yson.convert.yson_to_json(raw_schema)["$value"]
        for dict in data:
            for key in dict:
                value = dict[key]
                dict[key] = {'true':True,'false':False}.get(value, value)
        del data[0]
        data[0]['sort_order'] = 'ascending'
        return data

    def _fetch_data(self):
        logger.info("Start fetching data ST")
        query = 'st_issues.key as key, st_issues.resolved as resolved, ' \
                'st_issues.fixVersions as fixVersions, ' \
                'st_issues.customFields as customFields, ' \
                'st_issues.sla as sla, st_issues.created as created, ' \
                'st_types.key as type, ' \
                'st_resolutions.key as resolution ' \
                'FROM [{}] as st_issues ' \
                'JOIN [{}] as st_types ' \
                'ON st_issues.type = st_types.id ' \
                'LEFT JOIN [{}] as st_resolutions ' \
                'ON st_issues.resolution = st_resolutions.id'.format(
                self.source_path,
                "//home/startrek/tables/prod/yandex-team/common/types",
                "//home/startrek/tables/prod/yandex-team/common/resolutions"
        )

        yt_data = self.source_client.select_rows(query,
            format=yt.JsonFormat(attributes={"encode_utf8": False}))

        return list(yt_data)

    def _fetch_data_st_fixvers(self):
        logger.info("Start fetching data ST")
        yt_data = self.target_client.select_rows(
            'id, shortId from [{}]"'.format(
                "//home/helpdesk/cmdb/st_versions"),
            format=yt.JsonFormat(attributes={"encode_utf8": False}))
        data = {x["id"]:x["shortId"] for x in yt_data}
        return data

    def _processCustomFields(self, item):
        customfields = item["customFields"]
        del item["customFields"]

        abcserviceid = customfields.get('abcService')
        item["abcservice"] = abcserviceid[0]["id"] if abcserviceid else 0

    def _processFixVersions(self, item):
        fixver = item["fixVersions"]
        del item["fixVersions"]
        item["fixVersions"] = self.dict_ver.get(fixver[0]) if fixver else 0

    def _processSla(self, item):
        sla_info = item["sla"]
        del item["sla"]
        if not sla_info:
            return
        for sla_id in sla_info:
            counter = sla_info[sla_id]
            sla_sett_id = counter["settings"]["id"]
            if sla_sett_id not in [604,605]: continue
            sla_spent = counter.get("spent",0)
            sla_status = counter["violationStatus"]
            item["sla_{}_spent".format(sla_sett_id)] = sla_spent
            item["sla_{}_status".format(sla_sett_id)] = sla_status

    def _translate_data(self, data):
        logger.info("Start translating data")
        for item in data:
            self._processCustomFields(item)
            self._processSla(item)
            self._processFixVersions(item)
        logger.info("{}".format(len(data)))
        return data

    def main(self):
        pre_data = self._fetch_data()
        data = self._translate_data(pre_data)
        logger.info("Start upload to YT")
        self.write_data_to_yt(data)

@exc_thread_wrapper
def sync_st_to_static():
    STStaticSyncer(target_path='monitoring/st_issues_static',
                  target_client=YT_HAHN_CLIENT,
                  source_client=YT_MARKOV_CLIENT,
                  source_path='//home/startrek/tables/prod/yandex-team/queue/HDRFS/issues',
                  is_table_dynamic=False,
                  static_append=False,
                  recreate=False
                  ).main()
