import requests

from source.config import *
from source.monitoring.monitoring import SimpleYTBasedCollector
from source.utils import exc_thread_wrapper, OtherUtils

logger = logging.getLogger(__name__)


class StaffOfficesYTCollector(SimpleYTBasedCollector):
    def __init__(self):
        super().__init__(yt_table_name='cmdb/staff_all/offices', recreate=False,
                         client=YT_HAHN_CLIENT)
        self.schema = [
            {'name': 'id', 'type': 'int64',
             'sort_order': 'ascending'},
            {'name': 'country', 'type': 'string'},
            {'name': 'city', 'type': 'string'},
            {'name': 'is_city_deleted', 'type': 'boolean'},
            {'name': 'code', 'type': 'string'},
            {'name': 'address', 'type': 'string'},
            {'name': 'office_name', 'type': 'string'},
        ]

    def _fetch_data(self):
        result = []
        url = 'https://staff-api.yandex-team.ru/v3/offices'

        r = requests.get(url=url, headers=AUTH_HEADERS_TOOLS)
        count_pages = int(r.json()['pages']) + 1
        utiles = OtherUtils()

        search_fields = {'id':'id',
                         'country':'city.country.name.ru',
                         'city':'city.name.ru',
                         'is_city_deleted':'city.is_deleted',
                         'code':'code',
                         'address':'address.ru',
                         'office_name':'name.ru'
                         }

        for i in range(1, count_pages):
            logger.info('Fetch page # {}'.format(i))
            url = 'https://staff-api.yandex-team.ru/v3/offices?_fields={fields}&_page={page}'.format(
                page=i,
                fields = ','.join(search_fields.values())
            )
            r = requests.get(url=url, headers=AUTH_HEADERS_TOOLS)
            raw_data = r.json()['result']
            for item in raw_data:
                staff_d = utiles._extract_data_from_raw_data(item, search_fields)

                approved_results = {x:staff_d[x] for x in staff_d if x in search_fields.keys()}
                result.append(approved_results)

        return result

    def main(self):
        data = self._fetch_data()
        self.check_or_create_table()
        self.write_data_to_yt(data)

@exc_thread_wrapper
def main_staff_offices():
    StaffOfficesYTCollector().main()