import requests

from source.config import *
from source.monitoring.monitoring import SimpleYTBasedCollector
from source.utils import OtherUtils, exc_thread_wrapper

logger = logging.getLogger(__name__)

class StaffYTSync(SimpleYTBasedCollector):
    def __init__(self):
        super().__init__(yt_table_name='cmdb/staff', recreate=False, client=YT_HAHN_CLIENT)
        self.schema = [
            {'name': 'login', 'type': 'string',
             'sort_order': 'ascending'},
            {'name': 'is_dismissed', 'type': 'boolean'},
            {'name': 'is_trainee', 'type': 'boolean'},
            {'name': 'affilation', 'type': 'string'},
            {'name': 'bu', 'type': 'string'},
            {'name': 'dep_path', 'type': 'string'},
            {'name': 'location', 'type': 'string'},
            {'name': 'office_id', 'type': 'int64'},
            {'name': 'position', 'type': 'string'},
            {'name': 'join_at', 'type': 'string'},
            {'name': 'quit_at', 'type': 'string'},
            {'name': 'manager', 'type': 'string'},
            {'name': 'dep_slug', 'type': 'string'},
            {'name': 'work_phone', 'type': 'int64'},
        ]

    def _fetch_data(self):
        result = []
        headers = {
            "Authorization": 'OAuth {}'.format(
                OAUTH_TOOLS)
        }
        url = 'https://staff-api.yandex-team.ru/v3/persons?_fields=login,official.is_dismissed,official.affiliation,official.organization.name,department_group.ancestors.name,location.office.id,official.join_at,official.quit_at&_limit=1000&official.is_robot=false'

        r = requests.get(url=url, headers=headers)
        count_pages = int(r.json()['pages']) + 1
        utiles = OtherUtils()

        search_fields = {'login':'login',
                         'is_dismissed':'official.is_dismissed',
                         'affilation':'official.affiliation',
                         'position':'official.position.ru',
                         'bu':'official.organization.name',
                         'dep_path':'department_group.ancestors.name',
                         'dep_name':'department_group.name',
                         'loc_name':'location.office.name.ru',
                         'office_id':'location.office.id',
                         'join_at':'official.join_at',
                         'quit_at':'official.quit_at',
                         'heads':'department_group.department.heads',
                         'parent_heads':'department_group.parent.department.heads',
                         'dep_slug':'department_group.url',
                         'work_phone':'work_phone',
                         'is_trainee':'official.is_trainee'
                         }

        for i in range(1, count_pages):
            logger.info('Fetch page # {}'.format(i))
            url = 'https://staff-api.yandex-team.ru/v3/persons?_fields={fields}&_limit=1000&official.is_robot=false&_page={page}'.format(
                page=i,
                fields = ','.join(search_fields.values())
            )
            r = requests.get(url=url, headers=headers)
            raw_data = r.json()['result']
            for item in raw_data:
                staff_d = utiles._extract_data_from_raw_data(item, search_fields)
                full_dep_path = [x['name'] for x in staff_d["dep_path"]] + [staff_d["dep_name"]]
                manager = utiles.get_head_by_data(item['login'], staff_d['heads'], staff_d['parent_heads'])

                result.append({'login': item['login'],
                               'bu': staff_d['bu'],
                               'dep_path': '>'.join(full_dep_path),
                               'affilation': staff_d['affilation'],
                               'is_dismissed': staff_d['is_dismissed'],
                               'position': staff_d['position'],
                               'location': staff_d['loc_name'],
                               'office_id': staff_d['office_id'],
                               'quit_at': staff_d['quit_at'],
                               'join_at': staff_d['join_at'],
                               'dep_slug': staff_d['dep_slug'],
                               'manager': manager,
                               'work_phone': staff_d['work_phone'],
                               'is_trainee': staff_d['is_trainee']
                               })

        return result

    def main(self):
        logger.info("Start fetching data from Staff")
        data = self._fetch_data()
        logger.info("Start upload to YT")
        self.write_data_to_yt(data)

@exc_thread_wrapper
def main_staff_persons():
    logger.info('Start staff to YT sync')
    StaffYTSync().main()