import requests

from source.config import *
from source.monitoring.monitoring import SimpleYTBasedCollector
from source.utils import exc_thread_wrapper, OtherUtils

logger = logging.getLogger(__name__)


class StaffRoomsYTCollector(SimpleYTBasedCollector):
    def __init__(self):
        super().__init__(yt_table_name='cmdb/staff_all/rooms',
                         recreate=False,
                         client=YT_HAHN_CLIENT,
                         is_table_dynamic=False,
                         static_append=False)

        self.schema = [
            {'name': 'id', 'type': 'int64'},
            {'name': 'country', 'type': 'string'},
            {'name': 'city', 'type': 'string'},
            {'name': 'floor_name', 'type': 'string'},
            {'name': 'voice_conferencing', 'type': 'boolean'},
            {'name': 'video_conferencing', 'type': 'string'},
            {'name': 'seats', 'type': 'int64'},
            {'name': 'screen', 'type': 'int64'},
            {'name': 'projector', 'type': 'int64'},
            {'name': 'is_bookable', 'type': 'boolean'},
            {'name': 'name', 'type': 'string'},
            {'name': 'type', 'type': 'string'},
            {'name': 'office_name', 'type': 'string'},
        ]

    def _fetch_data(self):
        result = []
        base_url = 'https://staff-api.yandex-team.ru/v3/rooms'

        r = requests.get(url=base_url, headers=AUTH_HEADERS_TOOLS)
        count_pages = int(r.json()['pages']) + 1
        utiles = OtherUtils()
        dict_bool = {'false':False, 'true':True}
        search_fields = {'id':'id',
                         'country':'floor.office.city.country.name.ru',
                         'city':'floor.office.city.name.ru',
                         'floor_name':'floor.name.ru',
                         'video_conferencing':'equipment.video_conferencing',
                         'voice_conferencing':'equipment.voice_conferencing',
                         'seats':'equipment.seats',
                         'screen':'equipment.screen',
                         'projector':'equipment.projector',
                         'is_bookable':'is_bookable',
                         'name':'name.ru',
                         'type':'type',
                         'office_name':'floor.office.name.ru'}

        for i in range(1, count_pages):
            logger.info('Fetch page # {}'.format(i))
            url = '{base_url}?_fields={fields}&_page={page}'.format(
                page = i,
                fields = ','.join(search_fields.values()),
                base_url = base_url
            )
            r = requests.get(url=url, headers=AUTH_HEADERS_TOOLS)
            raw_data = r.json()['result']
            for item in raw_data:
                staff_d = utiles._extract_data_from_raw_data(item, search_fields)
                approved_results = {x: dict_bool.get(staff_d[x],staff_d[x]) for x in staff_d if x in search_fields.keys()}
                result.append(approved_results)

        return result

    def main(self):
        data = self._fetch_data()
        self.write_data_to_yt(data)

@exc_thread_wrapper
def main_staff_rooms():
    StaffRoomsYTCollector().main()