import logging

import yt.wrapper as yt
import copy
from source.config import *
from source.utils import (abc_help_logins, StartrekUtils)
from datetime import datetime, timedelta

logger = logging.getLogger(__name__)
YT_CLIENT = YT_VANGA_CLIENT
ST_MACROS_ID = 406 # COMP_LaptopMon (Мониторинг по использованию чужого ноутбука)

def find_assets_with_invalid_owner():
    """ Return instance_numbers where Jss login not equal OeBS login
        AND last check in one day ago
    """
    last_day = datetime.now() - timedelta(days = 1)
    str_day = last_day.strftime("%Y-%m-%d")
    logger.info('Search computers where Jss login not equal OeBS login')
    yt_data = YT_CLIENT.select_rows(
        'jss.instance_number, jss.serial_number , jss.active_user_sessions, jss.username, jss.last_check_in, \
        oebs.last_ticket_number, oebs.oebs_login, oebs.ext_login \
        FROM [{}] as jss \
        JOIN [{}] as oebs \
        ON jss.instance_number = oebs.instance_number \
        WHERE jss.username != "root" \
        AND jss.username != oebs.oebs_login \
        AND jss.username != oebs.ext_login \
        AND regex_partial_match("{}", jss.last_check_in)'.format(
            '//home/helpdesk/cmdb/jss', '//home/helpdesk/cmdb/notebooks', str_day
        ),
        format=yt.JsonFormat(attributes={"encode_utf8": False}))

    return list(yt_data)

def find_empty_oebs_login():
    """ Return instance_numbers where Jss login exists and no OeBS login
    """
    last_day = datetime.now() - timedelta(days = 1)
    str_day = last_day.strftime("%Y-%m-%d")
    logger.info('Search computers where OeBS login is empty')
    yt_data = YT_CLIENT.select_rows(
        'jss.instance_number, jss.serial_number , jss.username, jss.active_user_sessions, jss.last_check_in, \
        oebs.last_ticket_number, oebs.oebs_login, oebs.ext_login \
        FROM [{}] as jss \
        JOIN [{}] as oebs \
        ON jss.instance_number = oebs.instance_number \
        WHERE oebs.oebs_login = null \
        AND oebs.ext_login = "" \
        AND jss.username != "root" \
        AND jss.active_user_sessions != "" \
        AND regex_partial_match("{}", jss.last_check_in)'.format(
            '//home/helpdesk/cmdb/jss', '//home/helpdesk/cmdb/notebooks', str_day
        ),
        format=yt.JsonFormat(attributes={"encode_utf8": False}))

    return list(yt_data)

def issue_for_instance_exist(instance_number):
    """ Find issue for instance with wrong owner
    """
    filter_for_wrong_owner = 'Queue: HDRFS AND components: 52286 AND tags: "COMP_LaptopMon" AND tags:"inv:{}" AND Resolution: empty()'.format(instance_number)
    issue = st_client.issues.find(filter_for_wrong_owner)
    if issue:
        return True
    else:
        return False

def get_data_for_issue(instance_info, actions_for_laptop):
    """ Check what issue for inv not exist and create
        https://st.yandex-team.ru/TECHADMIN-1341
    """
    descr_dict = {
            'instance_number': instance_info['jss.instance_number'],
            'oebs_login': instance_info['oebs.oebs_login'],
            'oebs_ext_login': instance_info['oebs.ext_login'],
            'jss_login': instance_info['jss.username'],
            'jss_session': instance_info['jss.active_user_sessions'],
            'sn': instance_info['jss.serial_number'],
            'last_ticket': instance_info['oebs.last_ticket_number'],
            'last_check_in': instance_info['jss.last_check_in']
        }
        
    if instance_info['oebs.ext_login'] != '':
        descr_dict['oebs_ext_login'] = '{}@'.format(instance_info['oebs.ext_login'])

    jss_session = descr_dict['jss_session']
    if jss_session.strip() and jss_session != descr_dict['jss_login']:
        descr_dict['jss_login'] = '{}@ {}'.format(descr_dict['jss_login'], jss_session.strip())

    description = """
**Нужно проверить, что компьютер проведён правильно**
#|
|| Инвентарный номер | OeBS | OeBS (ext_login) | JSS logins | Serial Number | Последний тикет | Был в сети ||
|| !!Я.!!**{instance_number}** | {oebs_login}@ | {oebs_ext_login} | {jss_login}@ | %%{sn}%% | {last_ticket} | {last_check_in} ||
|#""".format(**descr_dict)
    actions_laptop = copy.deepcopy(actions_for_laptop)
    actions_laptop['tags'].append('inv:{}'.format(descr_dict['instance_number']))
    issue_data = {
                "summary":"Мониторинг по использованию чужого ноутбука (инв.{})".format(descr_dict['instance_number']),
                "description": description,
                "queue":"HDRFS",
                **actions_laptop
    }
    
    return issue_data

def main():

    actions_for_laptop_mon = StartrekUtils().get_st_macros_actions(ST_MACROS_ID)['Result']
    # All asserts with oebs_login != jss_login
    instance_with_wrong_login = find_assets_with_invalid_owner()
    helpdesk_logins = abc_help_logins()
    instances_on_help = [item for item in instance_with_wrong_login if item["oebs.oebs_login"] in helpdesk_logins]
    instances_without_oebs_login = find_empty_oebs_login()

    assets = instances_on_help + instances_without_oebs_login
    for asset in assets:
        issue_data = get_data_for_issue(asset, actions_for_laptop_mon)
        
        if not issue_for_instance_exist(asset['jss.instance_number']):
            try:
                logger.info('Try create issue for asset: inv: {}'.format(asset['jss.instance_number']))
                st_client.issues.create(**issue_data)
            except Exception as error:
                logger.error('Issues for asset on help not created. Error: {}'.format(error))
