# -*- coding: utf-8 -*-

import json
import logging
import requests
import time
from datetime import datetime
from startrek_client import Startrek, exceptions
from source.utils import StartrekUtils
from source.config import (OAUTH_TOOLS,
                    BOT_OAUTH,
                    MACROS_ID_KRAKEN)


class PassportAuth:
    """Авторизация на passport.yandex-team.ru"""
    def __init__(self, auth_login=None, auth_password=None, auth_cookies=None, auth_token=None):
        self.__login = auth_login
        self.__password = auth_password
        self.__cookies = auth_cookies
        self.__token = auth_token
        self.session = requests.Session()
        if self.__token:
            self.session.headers['Authorization'] = 'OAuth ' + self.__token
            result = self.session.get('https://passport.yandex-team.ru/')
            result.raise_for_status()
        else:
            logging.error(u'Укажите пару логин пароль или токен или куку')
            exit(0)


class BotCrm(PassportAuth):
    def get_request_top_users(self):
        request = self.session.get('https://bot.yandex-team.ru/api/v2/crm/person/top')
        request.raise_for_status()
        result = request.json()
        if result.get('error') is False:
            return tuple(result.get('result'))
        else:
            return False

    def get_request_crm_apps(self, **params):
        params_request = ''
        for key, value in params.items():
            params_request += key + '=' + value + '|'
        result = self.session.get('https://bot.yandex-team.ru/api/v2/crm/application/search/%s'
                                  % params_request).json()
        if result.get('error') is not False:
            logging.error(result.get('error'))
            return False
        else:
            return tuple(result.get('result'))

    def put_request_solved_app(self, app_id, app_text, res_text):
        put_request = {
            "application": app_text,
            "resolution": res_text,
            "action": "close"}
        request = self.session.put("https://bot.yandex-team.ru/api/v2/crm/application/id/%s" % app_id,
                                   data=json.dumps(put_request))
        request.raise_for_status()
        result = request.json()
        if result.get('error') is False:
            return result.get('result').get('stKey')
        else:
            logging.error(result.get('error'))
            return False

    def get_request_sip_users(self):
        request = self.session.get('https://bot.yandex-team.ru/api/sip_roles.php?instance=helpdesk')
        request.raise_for_status()
        request = request.json()
        result = {}
        for key in request.keys():
            role = request.get(key).get('role_string')
            agents_list = []
            try:
                for agent in request.get(key).get('agents'):
                    agents_list.append(agent.get('staff_login'))
            except TypeError:
                agents_list = False
            if agents_list:
                result[role] = agents_list
        return result

def create_app_text(date, roles):
    # формирование текста для application
    if roles == 'welcome' and date:
        return 'Визит в ХелпДеск %s' % datetime.fromtimestamp(int(date)).strftime(
            '%d %b %H:%M:%S')
    elif roles == 'income' and date:
        return 'Звонок в ХелпДеск %s' % datetime.fromtimestamp(int(date)).strftime(
            '%d %b %H:%M:%S')
    else:
        return False


def create_res_text(owner):
    # формирование текста для resolution
    if type(owner) is str:
        return '''Привет. У нас зафиксировано твое обращение
Подскажи, пожалуйста, удалось ли нам решить возникший вопрос?
Если да, подтверди решение этого тикета, нажав кнопку "Подтверждаю".
Если твой вопрос не был решен, нажми "Не подтверждаю" и призови {owner}@, мы вернемся к решению.
Если ты не нажмешь ни одну кнопку в течение 14 дней, тикет будет автоматически закрыт как подтвержденный.'''.format(owner=owner)
    elif type(owner) is list:
        return '''Привет. У нас зафиксировано твое обращение
Подскажи, удалось ли нам решить возникший вопрос?
Если да, подтверди решение этого тикета, нажав кнопку "Подтверждаю".
Если твой вопрос не был решен, нажми "Не подтверждаю" и призови {owner}@, мы вернемся к решению.
Если ты не нажмешь ни одну кнопку в течение 14 дней, тикет будет автоматически закрыт как подтвержденный.'''.format(owner='@ '.join(owner))
    else:
        return False


def check_date(date_target):
    # если апликуха создана текущим днем, функция возвращает False
    date_now = datetime.now().replace(microsecond=0)
    date_target = datetime.fromtimestamp(int(date_target))
    delta = (date_now.replace(hour=0, minute=0, second=0) - date_target.replace(hour=0, minute=0, second=0)).days
    if delta:
        return True
    else:
        return False

def main():
    logger = logging.getLogger(__name__)
    macros = StartrekUtils().get_st_macros_actions(MACROS_ID_KRAKEN)
    if macros['Error']:
        macros = {}
        logger.error(macros['Result'])
    else:
        macros = macros['Result']
        logger.info('Macros: ' + str(macros))
    try:
        logger.info('(KRAKEN) Start')
        userag = 'Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/30.0.1599.12785 YaBrowser/13.12.1599.12785 Safari/537.36'
        clientSt = Startrek(useragent=userag,
                            base_url='https://st-api.yandex-team.ru',
                            token=OAUTH_TOOLS)

        sessionCrm = BotCrm(auth_token=BOT_OAUTH)

        componentsDict = {'welcome': 'Welcome',
                          'income': 'Phone'
                          }
        locationDict = {'hd_avrora': 'helpdesk @ avrora',
                        'hd_msk': 'income @ msk',
                        'hd_workki': 'income @ workki'
                        }
        fixVersionsDict = {'hd_avrora': 'MSK Avrora',
                           'hd_msk': 'MSK Morozov',
                           'hd_workki': 'MSK Workki'
                           }
        assigneesDict = sessionCrm.get_request_sip_users()  # поулчаем логины залогиненх в телефониюю на инкам
        topsList = sessionCrm.get_request_top_users() # получаем лоигны ТОПов

        if topsList and assigneesDict:
            applicationsList = sessionCrm.get_request_crm_apps(status='OPEN',
                                                               location='hd_msk,hd_avrora,hd_workki',  # hd_msk hd_avrora
                                                               roles='income,welcome'  # income welcome
                                                               )
            for app in applicationsList:

                if not app.get('stKey') and app.get('Author') not in topsList and check_date(app.get('DateOpen')):
                    try:
                        appText = create_app_text(app.get('DateOpen'), app.get('Role'))
                        if app.get('Owner'):
                            resText = create_res_text(app.get('Owner'))
                        else:
                            resText = create_res_text(assigneesDict.get(locationDict.get(app.get('location_code'))))
                        if appText and resText:
                            stKey = sessionCrm.put_request_solved_app(app.get('Id'), appText, resText)
                            logger.info('(KRAKEN) Create ticket: {key}'.format(key=stKey))
                        else:
                            raise TypeError('ST message is empty')

                        checkCount = 3
                        while checkCount: # пушим данные в тикет
                            try:
                                updateField = {
                                    'assignee': app.get('Owner'),
                                    'followers': None if app.get('Owner') else assigneesDict.get(locationDict.get(app.get('location_code'))),
                                    'fixVersions': fixVersionsDict.get(app.get('location_code')),
                                    'channel': componentsDict.get(app.get('Role'))
                                }
                                clientSt.issues[stKey].transitions['resolved'].execute()
                                clientSt.issues[stKey].update(**{**macros, **updateField})
                                checkCount = False
                                logger.info('(KRAKEN) Update ticket: {key}'.format(key=stKey))
                            except exceptions.Conflict:
                                time.sleep(1)
                                checkCount -= 1
                    except requests.exceptions.HTTPError as error:
                        logging.error(u'(KRAKEN) ERR: %s' % error)
        else:
            raise TypeError(u'Top and Assignees list empty')
        logger.info(u'(KRAKEN) Finish')
    except Exception as error:
        logging.error(u'(KRAKEN) ERR: %s\n (KRAKEN) Stop' % error)
