# -*- coding: utf-8 -*-
import datetime
import functools
import json
import logging
import re
import smtplib
import time
from email.mime.text import MIMEText
from traceback import format_exc

import requests
from requests import Session
from requests.auth import HTTPBasicAuth
from zeep import Client
from zeep.transports import Transport

from source.config import *

logger = logging.getLogger(__name__)

class StatConnector(object):
    def __init__(self, name, title='Report Title'):
        self.name = name
        self.title = title
        self.auth_headers = {
            'StatRobotUser': STAT_USER,
            'StatRobotPassword': STAT_PWD}
        self.url = 'https://upload.stat.yandex-team.ru/_api'

    def typist(self,value):
        stat_type = {int: 'number', str: 'string', float: 'number'}.get(type(value))
        if stat_type:
            return stat_type
        else:
            raise Exception('Unsupported data type : {} for StatFace'.format(type(value)))

    def exists(self):
        data = requests.get('{url}/report/config?name={name}'.format(url=self.url,
                                                                     name=self.name),
                     headers=self.auth_headers)
        data_json = data.json()
        if data_json.get('_pk'):
            self.report_fields = [list(x.keys())[0] for x in data_json['fields']]
            self.user_config = {list(x.keys())[0]:x[list(x.keys())[0]] for x in data_json['user_config']['measures']}
            return True
        else:
            return False

    def create_table(self, data):
        measures = [{key:self.typist(data[0][key])} for key in data[0].keys() if key is not 'fielddate']
        report_config = {
            'dimensions': [
                {'fielddate': 'date'},
            ],
            'measures': measures,
        }
        data = requests.post(
            '{url}/report/config'.format(url=self.url),
            headers=self.auth_headers,
            data={
                'json_config': json.dumps(
                    {
                        'user_config': report_config,
                        'title': self.title,
                    }
                ),
                'name': self.name,
                'scale': 'd',
            },
        ).json()
        prefix = data.get('prefix')
        if prefix and re.match('Error.*',prefix):
            raise Exception('Problem with creation stat report {} : {}'.format(
                data['prefix'],
                data['message']
            ))

    def update_table(self, data):
        if hasattr(self,'report_fields') and data:
            remote_fields = self.report_fields
            data_fields = data[0].keys()
            remove_fields = [x for x in remote_fields if x not in data_fields and x != 'fielddate']
            added_fields = [x for x in data_fields if x not in remote_fields]
            if added_fields:
                for field in remove_fields:
                    field = field.lower()
                    field_type = self.user_config[field]
                    data[0][field] = {"number":0,
                                   "string":''}.get(field_type)
                self.create_table(data)

    def upload_data(self, data, scale='d'):
        date = datetime.datetime.now().strftime('%Y-%m-%d')

        if type(data) is not list:
            data = [data]

        if not self.exists():
            self.create_table(data)

        self.update_table(data)

        for item in data:
            if 'fielddate' not in item.keys():
                item['fielddate'] = date

        check = False
        try_count = 0
        while check == False and try_count < 5:
            try_count += 1
            try:
                r = requests.post(
                    '{url}/report/data'.format(url=self.url),
                    headers=self.auth_headers,
                    data={
                        'name': self.name,
                        'scale': scale,
                        'data': json.dumps({"values": data}),
                        '_append_mode': 0,
                    },
                )

                json1 = r.json()

                if len(json1['message']) == 72 or len(json1['message']) < 44:
                    logger.error((u'Status:', check, u'Table:',
                                   u'len content==72 or len content<44'))
                    logger.error(json1['message'])
                else:
                    check = True
                    logger.info('Upload succesfull {}'.format(self.name))
                    logger.info(('Status:', check, 'Table:', self.name, 'Lenght:',
                                  len(json1['message'])))
                    logger.info(json1['message'])

                time.sleep(3)
            except Exception as exc:
                logger.warning(exc)

class StatConnectorMoreDimensions(StatConnector):
    def create_table(self, data):
        measures = [{key:self.typist(data[0][key])} for key in data[0].keys() if key is not 'fielddate' and key is not 'location']
        report_config = {
            'dimensions': [
                {'fielddate': 'date'},
                {'location': 'string'}
            ],
            'measures': measures,
        }
        data = requests.post(
            '{url}/report/config'.format(url=self.url),
            headers=self.auth_headers,
            data={
                'json_config': json.dumps(
                    {
                        'user_config': report_config,
                        'title': self.title,
                    }
                ),
                'name': self.name,
                'scale': 'd',
            },
        ).json()
        prefix = data.get('prefix')
        if prefix and re.match('Error.*',prefix):
            raise Exception('Problem with creation stat report {} : {}'.format(
                data['prefix'],
                data['message']
            ))

class ClarityCollector(object):
    def __init__(self,mode = 'get'):
        self.user = os.environ['jamfuser']
        self.password = os.environ['jamfpsswd']
        self.mode = mode
        self.url = 'https://admin.clarity.yandex.net:8443/JSSResource/'

    def query(self,**kwargs):
        """Make query to Clarity api
        :param query: path to clarity api
        :param mode: type http mode for clarity api
        """
        if kwargs.get('mode') : self.mode = kwargs.get('mode')
        query = kwargs.get('query')

        req = {'get':requests.get,
               'post':requests.post,
               'put':requests.put}.get(self.mode)

        if not self.mode or not query:
            raise Exception('Uncorrect request')

        data = req(url='{url}{query}'.format(query=query,
                                              url=self.url),
                   auth=(self.user, self.password),
                   headers={'Accept': 'application/json'})
        return data.json()


class SerenityCollector(ClarityCollector):
    def __init__(self, **kwargs):
        super(SerenityCollector, self).__init__()
        self.url = kwargs.get('url','https://api.serenity.yandex.net:8443/JSSResource/')


class EmailLogger():

    def __init__(self, theme):
        self.theme = theme

    def send_email(self, recp, message):
        server = smtplib.SMTP('outbound-relay.yandex.net')
        msg = MIMEText(message, 'html', 'utf-8')
        msg['From'] = 'robot-help@yandex-team.ru'
        msg['Subject'] = self.theme
        if type(recp) is list:
            msg['To'] = ",".join(recp)
        else:
            msg['To'] = recp
        server.sendmail('robot-help@yandex-team.ru', recp,
                        msg.as_string())

def exc_thread_wrapper(func):
    @functools.wraps(func)
    def wrapper_count_calls(*args, **kwargs):
        logger = logging.getLogger('schedule')
        try:
            return func(*args, **kwargs)
        except:
            logger.error(format_exc())
    return wrapper_count_calls

def abc_help_logins():
    try:
        logins = []
        r = requests.get('https://abc-back.yandex-team.ru/api/v3/services/members/?service=1048&fields=person.login&page_size=1000', headers=AUTH_HEADERS_TOOLS)
        r.raise_for_status()
        logins = [item.get('person').get('login') for item in r.json().get('results')]
        logins.append('robot-bot-prod')
        logger.debug('ABC_Help logins: {logins}'.format(logins=logins))
    except Exception as err:
        logins = False
        logger.error(err)
    finally:
        return logins


def check_miracle_users_folder(login):
    """Check exist \\miracle\\Users\\<login>"""
    session = Session()
    session.verify = '/etc/ssl/certs/YandexInternalRootCA.pem'
    transport = Transport(session=session)
    session.auth = HTTPBasicAuth(MIRACLE_USER, MIRACLE_PASSWORD)
    try:
        client = Client(
            WSDL_URL,
            transport=transport)
        result = client.service.FolderExists(
        UserName = login)
    except Exception as error:
        logger.error('Cannon connect to {}'.format(WSDL_URL))
        raise Exception('Cannon connect to {}'.format(WSDL_URL))
    if result == 'True':
        return True
    elif result == 'False':
        return False
    else:
        logger.error(' Folder Check return not True or False')
        raise Exception('Return not True or False')

def get_staff_office_id_by_login(login):
    """Get staff office id for user"""
    staff_url_office = "{}persons?_pretty=0&_one=1&login={}&_fields=location.office.id".format(STAFF_URL, login)
    r = requests.get(staff_url_office, headers = AUTH_HEADERS_TOOLS)
    
    if r.status_code == 200:
        login_office = r.json()['location']['office']['id']
        if type(login_office) == int:
            logger.info('Staff api say office id: {} for User: {}'.format(login_office, login))
            return login_office
        else:
            logger.error('Not int in answer staff api for user: {}'.format(login))
            raise Exception('Not int in answer staff api for user: {}'.format(login))
    else:
        logger.error('Error api staff, HTTP code: {}'.format(r.status_code))
        raise Exception('Error api staff, HTTP code: {}'.format(r.status_code))


class StartrekUtils():
    def get_st_macros_actions(self, macros_id, queue = 'HDRFS'):
        """Get what changes make macros with task."""
        result = {}
        url = '{}queues/{}/macros/{}'.format(ST_URL, queue, macros_id)
        r = requests.get(url, headers = AUTH_HEADERS_TOOLS)

        if r.status_code == 200:
            data = r.json()
            if data.get("fieldChanges"):
                # Old type
                for item in data['fieldChanges']:
                    result[item['field']['id']] = item['value']
            else:
                # New type
                for item in data["issueUpdate"]:
                    for change_type in ["add", "set"]:
                        if item["update"].get(change_type):
                            result[item['field']['id']] = item["update"][
                                change_type]

            return {'Error': False, 'Result': result}

        elif r.status_code == 404:
            return {'Error': True, 'Result': 'Macros id: {} not found in queue: {}'.format(macros_id, queue)}
        elif r.status_code == 401:
            return {'Error': True, 'Result': 'Unauthorized request'}
        elif r.status_code == 400:
            return {'Error': True, 'Result':' Bad Request'}


class BotApiUtils():
    def fetch_osinfo(self, search_criteria, search_value, output_value):
        """function for use api osinfo
        https://wiki.yandex-team.ru/bot/api/osinfo/"""
        base_url = "https://bot.yandex-team.ru/api/osinfo.php?"
        base_url += "{}={}".format(search_criteria, search_value)
        base_url += "&output={}".format('|'.join(output_value))

        data = requests.get(base_url).json()

        return data if data["res"] == 1 else None

    def add_license(self, **kwargs):
        """
        :param kwargs:
        softwares : which license
        hardwares : which hardware
        :return:
        """
        softwares = kwargs.get('softwares')
        hardwares = kwargs.get('hadrwares')

        if not softwares or not hardwares:
            raise Exception('You mast select softwares and hardwares')

        error = True
        try_count = 2

        while error and try_count:
            data = json.dumps({"softwares": [softwares],
                               "quantity": 1,
                               "hardwares": [hardwares]})

            req = requests.post(
                "https://bot.yandex-team.ru/api/v2/oebs/software/relation",
                data=data,
                headers=AUTH_HEADERS_BOT)
            try_count -= 1
            error = req.json()["error"]
            if error:
                logger.warning(
                    '500 in bot api data: lic {} , inv: {}, response: {}'.format(
                        softwares, hardwares, req.content))
                time.sleep(2)
        if not error:
            return True
        return False

    def remove_license(self, **kwargs):
        """
        :param kwargs:
        softwares : which license
        hardwares : which hardware
        :return:
        """
        softwares = kwargs.get('softwares')
        hardwares = kwargs.get('hadrwares')
        login = kwargs.get('login')

        if not softwares or not (hardwares or login):
            logger.warning("{} {} {}".format(
                softwares, hardwares, login
            ))
            raise Exception('You mast select softwares and hardwares/login')

        error = True
        try_count = 2

        while error and try_count:
            pre_data = {"softwares": [softwares],
                               "quantity": 1,
                               "hardwares": [hardwares]}
            if login:
                pre_data["login"] = login
            if hardwares:
                pre_data["hardwares"] = [hardwares]

            data = json.dumps(pre_data)

            req = requests.delete(
                "https://bot.yandex-team.ru/api/v2/oebs/software/relation",
                data=data,
                headers=AUTH_HEADERS_BOT)
            try_count -= 1
            error = req.json()["error"]
            if error:
                logger.warning(
                    'Error in bot api data: lic {} , inv: {}, response: {}, {}'.format(
                        softwares, hardwares, req.json()["error"], req.status_code))
                time.sleep(2)
        if not error:
            return True
        return False

class OtherUtils():
    def fetch_trancievers_info(self):
        data = requests.get('http://noc-export.yandex.net/rt/inventory.json')
        return data.json()

    def check_miracle_users_folder(self, login):
        """Check exist \\miracle\\Users\\<login>"""
        session = Session()
        session.verify = '/etc/ssl/certs/YandexInternalRootCA.pem'
        transport = Transport(session=session)
        session.auth = HTTPBasicAuth(MIRACLE_USER, MIRACLE_PASSWORD)
        try:
            client = Client(
                WSDL_URL,
                transport=transport)
            result = client.service.FolderExists(
            UserName = login)
        except Exception as error:
            logger.error('Cannon connect to {}'.format(WSDL_URL))
            raise Exception('Cannon connect to {}'.format(WSDL_URL))
        if result == 'True':
            return True
        elif result == 'False':
            return False
        else:
            logger.error(' Folder Check return not True or False')
            raise Exception('Return not True or False')
    
    def login_is_dismissed(self, login):
        """ Check that the employee is fired on staff.
        """
        staff_url_dismiss = "{}persons?_pretty=0&_one=1&login={}&_fields=official.is_dismissed".format(STAFF_URL, login)
        r = requests.get(staff_url_dismiss, headers = AUTH_HEADERS_TOOLS)
        
        if r.status_code == 200:
            login_dismiss = r.json()['official']['is_dismissed']
            if type(login_dismiss) == bool:
                logger.info('Staff api say user: {} is dismissed = {}'.format(login, login_dismiss))
                return login_dismiss
            else:
                logger.error('Not bool in answer staff api for user: {}'.format(login))
                raise Exception('Not bool in answer staff api for user: {}'.format(login))
        else:
            logger.error('Error api staff, HTTP code: {}'.format(r.status_code))
            raise Exception('Error api staff, HTTP code: {}'.format(r.status_code))

    def _staff_get_attr(self, json, attr_list):
        if attr_list and isinstance(json, dict):
            attr = attr_list.pop(0)
            try:
                json = json[attr]
                return self._staff_get_attr(json, attr_list)
            except KeyError:
                logger.warning(
                    'Problem with parsing staff debug info :{} {}'.format(
                        json, attr
                    ))
                return None
        else:
            return json

    def _extract_data_from_raw_data(self, data, search_fields):
        result = {}
        for field in search_fields:
            search_path = search_fields[field].split('.')
            result[field] = self._staff_get_attr(data, search_path)

        logger.debug('Extract data from staff info: {}'.format(result))
        return result

    def extract_data_from_staff_by_login(self, login, search_fields):
        """Extract data from staff.
        :param login: user login
        :param search_fields: dict with search keys and path of the keys
        :type login: basestring
        :type search_fields: dict
        :return: dict with extracted data in format key:value
        :rtype: dict
        """


        url = "https://staff-api.yandex-team.ru/v3/persons?_pretty=1&_one=1&login={login}&_fields={fields}".format(
            login=login,
            fields=",".join([search_fields[x] for x in search_fields])
        )
        logger.info('Start fetch staff api {}')

        data = requests.get(url, headers={
            "Authorization": "OAuth {}".format(OAUTH_TOOLS)
        }).json()

        result = self._extract_data_from_raw_data(data, search_fields)
        logger.info('Extract data from staff info: {}'.format(result))

        return result

    def get_fix_version_by_office_id(self, office_id):
        """Get ST fixVersion id for Staff Office id."""
        fix_version = {}
        url_mapping = 'https://help.yandex-team.ru/back/api/v1/reserve/fix_versions/?format=json'
        r = requests.get(url_mapping, verify=False)
        settings_fixversion_list = r.json()
        fix_version_settings = [item for item in settings_fixversion_list if item["fixver_staff_id"] == office_id]
        if fix_version_settings:
            fix_version['fixVersions'] = [{'id': fix_version_settings[0]["fixver_st_id"]}]
            return fix_version
        else:
            fix_version['fixVersions'] = [{'id': u'53054'}] # 53054 == Homework
            return fix_version

    def get_head_by_login(self, login):
        """Get staff head by heads tree
        return head if user exist
        """
        head = None
        fields = {
            'head': 'department_group.department.heads.person.login',
            'tree_of_heads': 'groups.group.ancestors.department.heads.person.login'
        }
        staff_info = OtherUtils().extract_data_from_staff_by_login(login, fields)

        try:
            if staff_info['head'] is not None:
                head = staff_info['head'][0]['person']['login']
            else:
                raise KeyError
        except LookupError:
            try:
                tree_of_heads = staff_info['tree_of_heads'][0]['group']['ancestors']
            except TypeError:
                return None
            for index in range(len(tree_of_heads) - 1, 0, -1):
                if len(tree_of_heads[index]['department']['heads']) != 0:
                    head = tree_of_heads[index]['department']['heads'][0]['person']['login']
                    break
        return head

    def get_head_by_data(self, login, heads, parent_heads):
        head_list = [x['person']['login'] for x in heads if
                     x['role'] == 'chief']
        if head_list and head_list[0] != login:
            return head_list[0]
        else:
            if not parent_heads: return None
            head_list = [x['person']['login'] for x in parent_heads if
                         x['role'] == 'chief']
            if head_list:
                return head_list[0]

    def get_office_for_location(self, loc):
        """ Find fixVersion and currentOffice for OeBS location
        """
        fix_version = {}
        url_mapping = 'https://help.yandex-team.ru/back/api/v1/reserve/fix_versions/?format=json'
        r = requests.get(url_mapping, verify=False)
        settings_fixversion_list = r.json()
        fix_version_settings = [item for item in settings_fixversion_list if item["oebs_path"] == loc]
        if fix_version_settings:
            fix_version['fixVersions'] = [{'id': fix_version_settings[0]["fixver_st_id"]}]
            fix_version['currentOffice'] = fix_version_settings[0]["fixver_staff_id"]
            return fix_version
        else:
            fix_version['fixVersions'] = [{'id': u'53054'}]
            return fix_version


class StaffUtils():
    def groupmembership_by_group_url(self, url):
        data = requests.get('https://staff-api.yandex-team.ru/v3/groupmembership?_pretty=1&_fields=person.login&group.url={}'.format(url),
                            headers=AUTH_HEADERS_TOOLS).json()["result"]
        return [x["person"]["login"] for x in data]
