import requests
from source.config import OAUTH_TOOLS, BOT_OAUTH, st_client
import time
from startrek_client import exceptions
import logging
from source.utils import exc_thread_wrapper

logger = logging.getLogger(__name__)
templ_withvpn = "Привет! Необходимые для работы с VPN доступы уже есть. Настроить VPN можно по инструкции:\n" \
                "* Для корпоративных ноутбуков с Windows: https://download.yandex.ru/wiki/EXT/winopenvpnstandart.htm\n" \
                "* Для корпоративных ноутбуков с MacOS: https://download.yandex.ru/wiki/EXT/macopenvpn.htm\n" \
                "* Для VPN с USB-токеном в Windows: https://download.yandex.ru/wiki/EXT/winopenvpnrutocken.htm\n" \
                "* Для VPN с USB-токеном в MacOS: https://download.yandex.ru/wiki/EXT/macopenvpnrutocken.htm"
templ_withoutvpn = "Привет! Как руководитель сотрудника, подтвердите, пожалуйста, выдачу ему доступа к " \
                   "корпоративной сети по VPN"
templ_resolved = "Пожалуйста, подтверди решение этого тикета, нажав кнопку \"Подтверждаю\".\n" \
                 "Если твой вопрос не был решен, нажми \"Не подтверждаю\", и мы вернемся к решению.\n" \
                 "Если ты не нажмешь ни одну кнопку в течение 14 дней, тикет будет автоматически закрыт " \
                 "как подтвержденный."


class Client:

    def __init__(self, token=None):

        self.header = {
            "Content-Type" : "application/json",
            "Authorization": "OAuth %s" % token,
        }

    def get_data(self, url):
        for i in range(3):
            try:
                resp = requests.get(url, headers=self.header, timeout=10, verify=False)
                return resp.json()
            except Exception as e:
                time.sleep(2)
        raise NameError('No data found')


class UserInfo(Client):

    def get_staff(self, creator):

        url = "https://staff-api.yandex-team.ru/v3/persons?login=%s&_fields=official," \
              "department_group.ancestors,department_group.department" % creator
        try:
            fromstaf = self.get_data(url)['result'][0]
        except Exception as error:
            raise RuntimeError('Can\'t get info from staff with message %s' % error)
        else:
            return fromstaf

    def get_user_groups(self, login):

        url = "https://api.hd.yandex-team.ru/api/internal/crm/ldap/user/%s" % login
        try:
            userinfo = self.get_data(url)
        except Exception as error:
            raise RuntimeError('BOT-api error with message %s' % error)
        else:
            try:
                groups = userinfo['shortGroups']
            except KeyError:
                logger.error('Can\'t find shortGroups in bot-api answer')
                return []
            else:
                return groups


class DataCollect:

    def get_chef(self, creator: str, department_group: dict):

        if type(creator) is not str:
            raise TypeError('creator should be a string')
        if type(department_group) is not dict:
            raise TypeError('department_group should be a dict')
        for dpch in department_group['department']['heads']:
            if dpch['role'] == 'chief':
                break
        if creator != dpch['person']['login']:
            chef = dpch['person']['login']
        else:
            prevlevel = department_group['department']['level']-1
            for ancch in department_group['ancestors'][prevlevel]['department']['heads']:
                if ancch['role'] == 'chief':
                    chef = ancch['person']['login']
                    break
        return chef

    def vpn_group_search(self, groups: list):

        if type(groups) is not list:
            raise TypeError('groups should be a list')
        for gr in groups:
            if 'Office.VPN' in gr:
                return True
        return False


def with_vpn(task):

    task.transitions['inProgress'].execute()
    time.sleep(1)
    task.comments.create(text=templ_withvpn)
    time.sleep(1)
    task.transitions['resolved'].execute()
    time.sleep(1)
    task.comments.create(text=templ_resolved)


def process_issue(task):
    staf = UserInfo(token=OAUTH_TOOLS)
    bot = UserInfo(token=BOT_OAUTH)
    dc = DataCollect()
    try:
        creator = task.createdBy.id
        stf = staf.get_staff(creator)
        chef = dc.get_chef(creator, stf['department_group'])
        groups = bot.get_user_groups(creator)
        if groups != []:
            vpngroup = dc.vpn_group_search(groups)
            if vpngroup is True:
                with_vpn(task)
                logger.info('Task of %s was closed' % creator)
            else:
                task.transitions['inProgress'].execute()
                task.comments.create(summonee=chef, text=templ_withoutvpn)
                logger.info('Chief of %s was summoned' % creator)
    except Exception as error:
        logger.exception('%s with message %s' % (type(error), error))

@exc_thread_wrapper
def main():

    st = st_client
    try:
        vpn_tasks = st.issues.find(filter={'resolution':'empty()',
                                           'queue': 'HDRFS',
                                           'tags': 'addVpnOFR',
                                           'assignee': 'empty()'})
    except exceptions.StartrekError as error:
        logger.exception('StarTrek error with message %s' % error)
    else:
        for task in vpn_tasks:
            process_issue(task)

if __name__ == '__main__':
    main()
