#pragma once

#include <kernel/geodb/protos/geodb.pb.h>

#include <kernel/search_types/search_types.h>

#include <library/cpp/containers/dense_hash/dense_hash.h>
#include <library/cpp/langs/langs.h>

#include <util/ysaveload.h>
#include <util/generic/ptr.h>
#include <util/generic/set.h>
#include <util/generic/vector.h>
#include <util/generic/yexception.h>

#include <functional>

/** GeoDataBase.
 * <p>Loads data from protobuf which can be generated by tools/geodb_loader.
 * <p>Some info about fields you can find on wiki: https://wiki.yandex-team.ru/Libgeobase/http-export
 */
namespace NGeoDB {
    class TError : public yexception {
    };
    class TInvalidVersion : public TError {
    };

    class TGeoKeeper;
    using TGeoKeeperHolder = THolder<TGeoKeeper>;

    /** It represents an adapeter for protobuf data TRegionProto by
     * introducing additional functions which helps you to navigate through
     * geo hierarchy in a more simple way!
     * <p>All the protobuf data is accessable by Get() or operator->() methods.
     * @see TGeoKeeper
     */
    class TGeoPtr {
    public:
        /** Create an invalid geo object.
         * <p>It means that all IDs are END_CATEG (-1).
         * <p>Cast to bool is false.
         */
        TGeoPtr();

        /** Create a valid geo object based on protobuf data and the manager
         * which is the owner of all the regions.
         * <p>Strongly recommends not to create it by yourself and use TGeoKeeper::Find()
         * @see TGeoKeeper::Find()
         * @param[in] proto a ref to protobuf data
         * @param[in] geoKeeper a pointer to TGeoKeeper object where the real data is loaded
         */
        TGeoPtr(const TRegionProto& proto, const TGeoKeeper* geoKeeper);

        // smartptr-like methods
        /** Check if the object is valid or not.
         * @see TGeoPtr()
         * @return true if object is valid
         */
        explicit operator bool () const {
            return Proto_->GetId() != END_CATEG;
        }

        /** Cast to TCateg. Just for convenience.
         * @return Id of the current region or -1 (END_CATEG)
         */
        explicit operator TCateg () const {
            return Proto_->GetId();
        }

        /** Pretends to be a smartpointer. Get access to internal protobuf data.
         * @see operator->()
         * @return a pointer to the region's protobuf data (if nullptr in case it is not valid)
         */
        const TRegionProto* Get() const {
            return Proto_;
        }

        /** The same as Get().
         * @see Get()
         * @return  a pointer to the region's protobuf data (if nullptr in case it is not valid)
         */
        const TRegionProto* operator->() const {
            return Proto_;
        }

        /** Compare two TGeoPtr objects.
         * <p>Internally it compares the IDs of the objects.
         * @see operator!=()
         * @return true if IDs are equal
         */
        bool operator==(const TGeoPtr& rhs) const {
            return Get()->GetId() == rhs->GetId();
        }

        /** Compare two TGeoPtr objects.
         * @see operator==()
         * @return true if IDs are NOT equal
         */
        bool operator!=(const TGeoPtr& rhs) const {
            return Get()->GetId() != rhs->GetId();
        }

        // specific geo-hierarchy browsing logic

        /** Check if the checkRegion is one of our ancestors.
         * <p>For example:
         * @code
         * TGeoPtr russia = geodb->Find(225);
         * geodb->Find(213).IsIn(russia); // is true
         * geodb->Find(213).IsIn(geodb->Find(213)); // is also true!!!
         * @endcode
         * @see IsIn(TCateg)
         * @param[in] probableParent is a TGeoPtr object which we want to check if it is our ancestor.
         * @return true if this is one of our ancestor.
         */
        bool IsIn(const TGeoPtr& probableParent) const;

        /** The same as IsIn(const TGeoPtr&) but use not object but region ID (TCateg)
         * @code
         * geodb->Find(213).IsIn(213); // is true!!!
         * @endcode
         * @see IsIn(const TGeoPtr&)
         * @param[in] probableParentId is an region ID which we want to check if it is our ancestor.
         * @return true if this is one of our ancestor.
         */
        bool IsIn(TCateg probableParentId) const;

        /** Returns the nearest parent.
         * <p>i.e. for Moscow(213) it will be MosObl(1)
         * @see TRegionProto::GetParentId() if you want ID instead of object
         * @return parent object
         */
        TGeoPtr Parent() const;

        /** Return first ancestor which has specific type.
         * <p>For region Moscow(213) and type COUNTRY it is Russia(225).
         * <p>For world or continent it is an invalid region.
         * <p>
         * @see TRegionProto::GetType()
         * @param[in] type is a type of region you want to find.
         * @return the a valid or invalid region
         */
        TGeoPtr ParentByType(EType type) const;

        /**@{*/
        /** Prepare the list of disambiguated parents from the current region with using a userRegion.
         * This logic was ported from perl report
         * [YxWeb::Util::Region](https://arc.yandex-team.ru/wsvn/arc/trunk/arcadia/web/report/lib/YxWeb/Util/Region.pm?op=blame&rev=1757241#l1364)
         * @param[in] userRegion is the region where the user is!
         * @param[out] disambiguated is a destination list of disambiguated parents
         * @return true if the disambiguated list has been modified otherwise false
         */
        bool Disambiguate(const TGeoPtr& userRegion, TVector<TGeoPtr>& disambiguated) const;
        bool Disambiguate(const TGeoPtr& userRegion, TVector<TCateg>& disambiguated) const;
        /**@}*/

        /** Returns the country of the region.
         * <p>If the regions doesn't have country (for example it is continent) it returns invalid region.
         * @return country region
         */
        TGeoPtr Country() const;

        /** Returns the capital of the country.
         * <p>It returns capital only for regions which type is COUNTRY otherwise it will be invalid.
         * <p>If there is not capital or moreover no data about capital it returns an invalid TGeoPtr.
         * @see CountryCapital()
         * @return capital region
         */
        TGeoPtr Capital() const {
            return COUNTRY == Get()->GetType() ? CountryCapital() : TGeoPtr();
        }

        /** Return the capital of the region's country.
         * <p>It doesn't matter if the region is country or not. Tries to find country and then get its capital.
         * @see Capital()
         * @see TRegionProto::GetCountryCapitalId() return ID instead of an object.
         * @return capital region.
         */
        TGeoPtr CountryCapital() const;

        /** Return the administrative center of the current region.
         * <p>For countries it is capital.
         * @see TRegionProto::GetChiefId()
         * @return a valid region if it has one otherwise invalid
         */
        TGeoPtr Chief() const;

        /** Return struct of names for specific language in different cases
         * @param[in] lang is the lang id
         * @return the struct of names
         */
        const TRegionProto::TNames* FindNames(ELanguage lang) const;

        /** Return struct of names for specific language in different cases
         * @see LanguageByName()
         * @param[in] lang is the lang string
         * @return the struct of names
         */
        const TRegionProto::TNames* FindNames(const TStringBuf& lang) const;

    private:
        using TCheckRegionFunc = std::function<const TRegionProto*(const TRegionProto*)>;
        TGeoPtr ParentBy(TCheckRegionFunc predicate) const;

    private:
        TRegionProto const* Proto_;
        TGeoKeeper const* GeoKeeper_;
    };

    /** Loader for geobase data.
     * You can create data file by using tools/geodb_proto. In addition
     * there is a task in sandbox named MAKE_GEODB_DATA
     * <p>Examples:
     * @code
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(input);
        NGeoDB::TGeoPtr moscow = geodb->Find(213);
        if (moscow) {
            // cool!
            if (moscow == geodb->Find(54)) {
                // won't happened!!!
            }

            TCateg parentId = moscow->GetParentId();
            TGeoPtr parent = moscow.Parent();

            if (moscow->IsIn(225) || moscow->IsIn(geodb->Find(225))) {
                // cool!
            }
        }
     * @endcode
     * @see TGeoPtr
     */
    class TGeoKeeper {
    protected:
        using TRegionMap = TDenseHash<TCateg, TRegionProto>;
        TRegionMap Index_;       // TCateg -> TRegionProto
        ui32 Version_;

    public:
        using TConstIterator = TDenseHash<TCateg, TRegionProto>::const_iterator;

        /** Create an empty geo database.
         * @see Load(), LoadToHolder()
         */
        TGeoKeeper();

        /** Create geokeeper and populate protobuf TRegionProto data in one place!
         * @see Load()
         * @param[in] stream is a protobuf list which can be made by TGeoKeeper::Save()
         * @return a holder to TGeoKeeper
         */
        static TGeoKeeperHolder LoadToHolder(IInputStream& stream);

        /** Create geokeeper and populate protobuf TRegionProto data in one place!
         * @see Load()
         * @param[in] fileName with serialized TGeoKeeper
         * @return a holder to TGeoKeeper
         */
        static TGeoKeeperHolder LoadToHolder(const TString& fileName);

        /** Search region by id and return TGeoPtr.
         * <p>If there is not region data for requested id it returns an invalid TGeoPtr
         * @param[in] id is the region id you want to find!
         * @return a valid or invalid region
         */
        TGeoPtr Find(TCateg id) const;

        /** Checks whether geo database is empty.
         * @return true if database is empty.
         */
        bool Empty() const {
            return Index_.Empty();
        }

        /** Return a const iterator to hash of protobufs data
         * @return iterator
         */
        TConstIterator begin() const {
            return Index_.begin();
        }
        TConstIterator cbegin() const {
            return Index_.begin();
        }

        TConstIterator end() const {
            return Index_.end();
        }
        TConstIterator cend() const {
            return Index_.end();
        }

        inline void InsertRegion(const TRegionProto& region) {
            Index_[region.GetId()] = region;
        }

        void Save(IOutputStream* s) const;
        void Load(IInputStream* s);
    };

    TGeoPtr GetBestGeo(const TGeoPtr& userRegion, const TSet<TGeoPtr>& geos);
    TGeoPtr GetBestGeo(const TCateg userRegionId, const TSet<TCateg>& ids, const TGeoKeeper& geoDb);

    /** Needs for <b>arcadia style serialize</b> TRegionProto message. Created only to reduce
     * compile time by moving body of this function in cpp file (with messagext.h).
     * @see Save<NGeoDB::TRegionProto>(IOutputStream* output, const NGeoDB::TRegionProto& msg)
     */
    void SaveRegionProto(IOutputStream* output, const NGeoDB::TRegionProto& msg);
    /** Needs for <b>arcadia style deserialize</b> TRegionProto message. Created only to reduce
     * compile time by moving body of this function in cpp file (with messagext.h)
     * @see Load<NGeoDB::TRegionProto>(IInputStream* input, NGeoDB::TRegionProto& msg)
     */
    void LoadRegionProto(IInputStream* input, NGeoDB::TRegionProto& msg);

    /** Returns shortest distance in geotree between geoA and geoB
     */
    size_t GetShortestDistance(const TGeoPtr& geoA, const TGeoPtr& geoB) noexcept;

} // namespace NGeoDB

template <>
inline void Save<NGeoDB::TRegionProto>(IOutputStream* output, const NGeoDB::TRegionProto& msg) {
    NGeoDB::SaveRegionProto(output, msg);
}

template <>
inline void Load<NGeoDB::TRegionProto>(IInputStream* input, NGeoDB::TRegionProto& msg) {
    NGeoDB::LoadRegionProto(input, msg);
}


/** Actually it is needed for using with set/hash.
 * <p>It simply compare the ids of the two geoobjects
 * @param[in] l = the geo object/geoid which is compared
 * @param[in] r = the geo object/geoid which is compared with
 * @return true if l's geoid is less than r's one
 */
template<>
struct TLess<NGeoDB::TGeoPtr> {
    using is_transparent = void;

    bool operator()(const NGeoDB::TGeoPtr& l, const NGeoDB::TGeoPtr& r) const {
        return l->GetId() < r->GetId();
    }

    bool operator()(const NGeoDB::TGeoPtr& l, const TCateg& r) const {
        return l->GetId() < r;
    }

    bool operator()(const TCateg& l, const NGeoDB::TGeoPtr& r) const {
        return l < r->GetId();
    }
};
