#include "translocal.h"

#include <kernel/geodb/geodb.h>
#include <kernel/relev_locale/protos/serptld.pb.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/hash_set.h>
#include <util/generic/vector.h>

using NGeoDB::TranslocalParent;
using NGeoDB::TranslocalParentID;
using NGeoDB::TranslocalCountry;
using NGeoDB::TranslocalCountryID;

static const TString GEODB_PATH = "geodb.bin";

static constexpr TCateg RUSSIA = 225;
static constexpr TCateg UKRAINE = 187;
static constexpr TCateg BELARUS = 149;
static constexpr TCateg KAZAKHSTAN = 159;
static constexpr TCateg TURKEY = 983;
static constexpr TCateg FRANCE = 124;
static constexpr TCateg USA = 84;
static constexpr TCateg INDONESIA = 10095;
static constexpr TCateg GERMANY = 96;
static constexpr TCateg DUBNA = 215;  // city in Russia
static constexpr TCateg DUBNA_PARENT = 120992;  // Dubna Urban Okrug
static constexpr TCateg ODESSA = 145;  // city in Ukraine
static constexpr TCateg ODESSA_PARENT = 20541;  // Odessa District
static constexpr TCateg BREST = 153;  // city in Belarus
static constexpr TCateg BREST_PARENT = 29632;  // Brest District
static constexpr TCateg CHIMKENT = 221;  // city in Kazakhstan
static constexpr TCateg CHIMKENT_PARENT = 29417;  // South-Kazakhstan District
static constexpr TCateg ADANA = 11501;  // city in Turkey
static constexpr TCateg ADANA_PARENT = 103669;  // Adana (province)
static constexpr TCateg NANCY = 109137;  // city in Frace
static constexpr TCateg NANCY_PARENT = 104369; // Lorraine
static constexpr TCateg DRESDEN = 10407;  // city in Germany
static constexpr TCateg DRESDEN_PARENT = 103759;  // Saxony
static constexpr TCateg MEDAN = 10577;  // city in Indonesia
static constexpr TCateg MEDAN_PARENT = 120796;  // North Sumatra
static constexpr TCateg LOUISVILLE = 21190;  // city in USA
static constexpr TCateg LOUISVILLE_PARENT = 29334;  // Commonwealth of Kentucky
static constexpr TCateg KHAMOVNIKI_DISTRICT = 120542;  // district of Central Administrative Okrug of the federal city of Moscow, Russia
static constexpr TCateg MOSCOW = 213;  // city in Russia

static constexpr TCateg REPUBLIC_OF_CRIMEA = 977;
static constexpr TCateg CRIMEAN_FEDERAL_DISTRICT = 115092;
static constexpr TCateg SEVASTOPOL = 959; // city in Crimea

template <>
void Out<NGeoDB::ETranslocalityHack>(
    IOutputStream& out,
    TTypeTraits<NGeoDB::ETranslocalityHack>::TFuncParam value) {
    switch (value) {
        case NGeoDB::NO_HACK: {
            out << "no";
            [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
        } case NGeoDB::CRIMEA_HACK: {
            out << "crimea";
            break;
        }
    }
}

namespace {
    struct TIDAndParent {
        TCateg ID;
        TCateg Parent;
    };

    struct TIDAndCountry {
        TCateg ID;
        TCateg Country;
    };
}

static const TVector<TCateg> COUNTRIES = {
    RUSSIA,
    UKRAINE,
    BELARUS,
    KAZAKHSTAN,
    TURKEY,
    FRANCE,
    GERMANY,
    INDONESIA,
    USA
};

static const TVector<TIDAndParent> ID_AND_PARENT = {
    {DUBNA, DUBNA_PARENT},
    {ODESSA, ODESSA_PARENT},
    {BREST, BREST_PARENT},
    {CHIMKENT, CHIMKENT_PARENT},
    {ADANA, ADANA_PARENT},
    {NANCY, NANCY_PARENT},
    {DRESDEN, DRESDEN_PARENT},
    {MEDAN, MEDAN_PARENT},
    {LOUISVILLE, LOUISVILLE_PARENT}
};

static const TVector<TIDAndCountry> ID_AND_COUNTRY = {
    {DUBNA, RUSSIA},
    {ODESSA, UKRAINE},
    {BREST, BELARUS},
    {CHIMKENT, KAZAKHSTAN},
    {ADANA, TURKEY},
    {NANCY, FRANCE},
    {DRESDEN, GERMANY},
    {MEDAN, INDONESIA},
    {LOUISVILLE, USA}
};

static TVector<TCateg> GetAllKnownIDs() {
    auto ids = THashSet<TCateg>{};
    for (const auto& value : ID_AND_PARENT) {
        ids.insert(value.ID);
        ids.insert(value.Parent);
    }

    for (const auto& value : ID_AND_COUNTRY) {
        ids.insert(value.ID);
        ids.insert(value.Country);
    }

    ids.insert(REPUBLIC_OF_CRIMEA);
    ids.insert(CRIMEAN_FEDERAL_DISTRICT);
    ids.insert(SEVASTOPOL);

    return {ids.cbegin(), ids.cend()};
}

static const auto ALL_KNOWN_IDS = GetAllKnownIDs();

static TVector<EYandexSerpTld> GetAllRelevantTLDs() {
    auto tlds = TVector<EYandexSerpTld>{};
    for (auto i = 0; i < EYandexSerpTld_ARRAYSIZE; ++i) {
        if (!EYandexSerpTld_IsValid(i)) {
            continue;
        }

        const auto tld = static_cast<EYandexSerpTld>(i);
        if (YST_UNKNOWN == tld) {
            continue;
        }

        tlds.push_back(tld);
    }

    return tlds;
}

static const auto RELEVANT_TLDS = GetAllRelevantTLDs();

static void CheckTranslocalParentConsistency(const TCateg id, const EYandexSerpTld tld,
                                             const NGeoDB::TGeoKeeper& geodb) {
    const auto idWithData = geodb.Find(id);
    auto hackOne = NGeoDB::ETranslocalityHack{};
    const auto parentOne = TranslocalParent(idWithData, tld, geodb, hackOne)->GetId();
    const auto parentTwo = TranslocalParent(idWithData, tld, geodb)->GetId();

    auto hackTwo = NGeoDB::ETranslocalityHack{};
    const auto parentThree = TranslocalParent(id, tld, geodb, hackTwo)->GetId();
    const auto parentFour = TranslocalParent(id, tld, geodb)->GetId();

    auto hackThree = NGeoDB::ETranslocalityHack{};
    const auto parentFive = TranslocalParentID(idWithData, tld, geodb, hackThree);
    const auto parentSix = TranslocalParentID(idWithData, tld, geodb);

    auto hackFour = NGeoDB::ETranslocalityHack{};
    const auto parentSeven = TranslocalParentID(id, tld, geodb, hackFour);
    const auto parentEight = TranslocalParentID(id, tld, geodb);

    UNIT_ASSERT_VALUES_EQUAL(parentOne, parentTwo);
    UNIT_ASSERT_VALUES_EQUAL(parentOne, parentThree);
    UNIT_ASSERT_VALUES_EQUAL(parentOne, parentFour);
    UNIT_ASSERT_VALUES_EQUAL(parentOne, parentFive);
    UNIT_ASSERT_VALUES_EQUAL(parentOne, parentSix);
    UNIT_ASSERT_VALUES_EQUAL(parentOne, parentSeven);
    UNIT_ASSERT_VALUES_EQUAL(parentOne, parentEight);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackTwo);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackThree);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackFour);
}

static void CheckTranslocalCountryConsistency(const TCateg id, const EYandexSerpTld tld,
                                              const NGeoDB::TGeoKeeper& geodb) {
    const auto idWithData = geodb.Find(id);
    auto hackOne = NGeoDB::ETranslocalityHack{};
    const auto countryOne = TranslocalCountry(idWithData, tld, geodb, hackOne)->GetId();
    const auto countryTwo = TranslocalCountry(idWithData, tld, geodb)->GetId();

    auto hackTwo = NGeoDB::ETranslocalityHack{};
    const auto countryThree = TranslocalCountry(id, tld, geodb, hackTwo)->GetId();
    const auto countryFour = TranslocalCountry(id, tld, geodb)->GetId();

    auto hackThree = NGeoDB::ETranslocalityHack{};
    const auto countryFive = TranslocalCountryID(idWithData, tld, geodb, hackThree);
    const auto countrySix = TranslocalCountryID(idWithData, tld, geodb);

    auto hackFour = NGeoDB::ETranslocalityHack{};
    const auto countrySeven = TranslocalCountryID(id, tld, geodb, hackFour);
    const auto countryEight = TranslocalCountryID(id, tld, geodb);

    UNIT_ASSERT_VALUES_EQUAL(countryOne, countryTwo);
    UNIT_ASSERT_VALUES_EQUAL(countryOne, countryThree);
    UNIT_ASSERT_VALUES_EQUAL(countryOne, countryFour);
    UNIT_ASSERT_VALUES_EQUAL(countryOne, countryFive);
    UNIT_ASSERT_VALUES_EQUAL(countryOne, countrySix);
    UNIT_ASSERT_VALUES_EQUAL(countryOne, countrySeven);
    UNIT_ASSERT_VALUES_EQUAL(countryOne, countryEight);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackTwo);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackThree);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackFour);
}

static void CheckTranslocalIsInConsistency(const TCateg child, const TCateg probableParent,
                                           const EYandexSerpTld tld,
                                           const NGeoDB::TGeoKeeper& geodb) {
    const auto childPtr = geodb.Find(child);
    const auto probableParentPtr = geodb.Find(probableParent);

    auto hackOne = NGeoDB::ETranslocalityHack{};
    const auto isInOne = TranslocalIsIn(childPtr, probableParentPtr, tld, geodb, hackOne);
    const auto isInTwo = TranslocalIsIn(childPtr, probableParentPtr, tld, geodb);

    auto hackTwo = NGeoDB::ETranslocalityHack{};
    const auto isInThree = TranslocalIsIn(child, probableParent, tld, geodb, hackTwo);
    const auto isInFour = TranslocalIsIn(child, probableParent, tld, geodb);

    auto hackThree = NGeoDB::ETranslocalityHack{};
    const auto isInFive = TranslocalIsIn(childPtr, probableParent, tld, geodb, hackThree);
    const auto isInSix = TranslocalIsIn(childPtr, probableParent, tld, geodb);

    auto hackFour = NGeoDB::ETranslocalityHack{};
    const auto isInSeven = TranslocalIsIn(child, probableParentPtr, tld, geodb, hackFour);
    const auto isInEight = TranslocalIsIn(child, probableParentPtr, tld, geodb);

    UNIT_ASSERT_VALUES_EQUAL(isInOne, isInTwo);
    UNIT_ASSERT_VALUES_EQUAL(isInOne, isInThree);
    UNIT_ASSERT_VALUES_EQUAL(isInOne, isInFour);
    UNIT_ASSERT_VALUES_EQUAL(isInOne, isInFive);
    UNIT_ASSERT_VALUES_EQUAL(isInOne, isInSix);
    UNIT_ASSERT_VALUES_EQUAL(isInOne, isInSeven);
    UNIT_ASSERT_VALUES_EQUAL(isInOne, isInEight);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackTwo);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackThree);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackFour);
}

static void CheckTranslocalParentExcludingTypesConsistency(
    const TCateg child, const EYandexSerpTld tld, const NGeoDB::TGeoKeeper& geodb,
    const TVector<NGeoDB::EType>& regionTypesToExclude
){
    const auto childPtr = geodb.Find(child);

    auto hackOne = NGeoDB::ETranslocalityHack{};
    const auto parentOne = TranslocalParentExcludingTypes(childPtr, tld, geodb,
                                                          regionTypesToExclude, hackOne);
    const auto parentTwo = TranslocalParentExcludingTypes(childPtr, tld, geodb,
                                                          regionTypesToExclude);

    auto hackTwo = NGeoDB::ETranslocalityHack{};
    const auto parentThree = TranslocalParentExcludingTypes(child, tld, geodb, regionTypesToExclude,
                                                            hackTwo);
    const auto parentFour = TranslocalParentExcludingTypes(child, tld, geodb, regionTypesToExclude);

    auto hackThree = NGeoDB::ETranslocalityHack{};
    const auto parentFive = TranslocalParentIDExcludingTypes(childPtr, tld, geodb,
                                                             regionTypesToExclude, hackThree);
    const auto parentSix = TranslocalParentIDExcludingTypes(childPtr, tld, geodb,
                                                            regionTypesToExclude);

    auto hackFour = NGeoDB::ETranslocalityHack{};
    const auto parentSeven = TranslocalParentIDExcludingTypes(child, tld, geodb,
                                                              regionTypesToExclude, hackFour);
    const auto parentEight = TranslocalParentIDExcludingTypes(child, tld, geodb,
                                                              regionTypesToExclude);

    UNIT_ASSERT_VALUES_EQUAL(parentOne->GetId(), parentTwo->GetId());
    UNIT_ASSERT_VALUES_EQUAL(parentOne->GetId(), parentThree->GetId());
    UNIT_ASSERT_VALUES_EQUAL(parentOne->GetId(), parentFour->GetId());
    UNIT_ASSERT_VALUES_EQUAL(parentOne->GetId(), parentFive);
    UNIT_ASSERT_VALUES_EQUAL(parentOne->GetId(), parentSix);
    UNIT_ASSERT_VALUES_EQUAL(parentOne->GetId(), parentSeven);
    UNIT_ASSERT_VALUES_EQUAL(parentOne->GetId(), parentEight);

    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackTwo);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackThree);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackFour);
}

static void CheckTranslocalReduceExcludingTypesConsistency(
    const TCateg child, const EYandexSerpTld tld, const NGeoDB::TGeoKeeper& geodb,
    const TVector<NGeoDB::EType>& regionTypesToExclude
){
    const auto childPtr = geodb.Find(child);

    auto hackOne = NGeoDB::ETranslocalityHack{};
    const auto regionOne = TranslocalReduceExcludingTypes(childPtr, tld, geodb,
                                                          regionTypesToExclude, hackOne);
    const auto regionTwo = TranslocalReduceExcludingTypes(childPtr, tld, geodb,
                                                          regionTypesToExclude);

    auto hackTwo = NGeoDB::ETranslocalityHack{};
    const auto regionThree = TranslocalReduceExcludingTypes(child, tld, geodb, regionTypesToExclude,
                                                            hackTwo);
    const auto regionFour = TranslocalReduceExcludingTypes(child, tld, geodb, regionTypesToExclude);

    auto hackThree = NGeoDB::ETranslocalityHack{};
    const auto regionFive = TranslocalReduceIDExcludingTypes(childPtr, tld, geodb,
                                                             regionTypesToExclude, hackThree);
    const auto regionSix = TranslocalReduceIDExcludingTypes(childPtr, tld, geodb,
                                                            regionTypesToExclude);

    auto hackFour = NGeoDB::ETranslocalityHack{};
    const auto regionSeven = TranslocalReduceIDExcludingTypes(child, tld, geodb,
                                                              regionTypesToExclude, hackFour);
    const auto regionEight = TranslocalReduceIDExcludingTypes(child, tld, geodb,
                                                              regionTypesToExclude);

    UNIT_ASSERT_VALUES_EQUAL(regionOne->GetId(), regionTwo->GetId());
    UNIT_ASSERT_VALUES_EQUAL(regionOne->GetId(), regionThree->GetId());
    UNIT_ASSERT_VALUES_EQUAL(regionOne->GetId(), regionFour->GetId());
    UNIT_ASSERT_VALUES_EQUAL(regionOne->GetId(), regionFive);
    UNIT_ASSERT_VALUES_EQUAL(regionOne->GetId(), regionSix);
    UNIT_ASSERT_VALUES_EQUAL(regionOne->GetId(), regionSeven);
    UNIT_ASSERT_VALUES_EQUAL(regionOne->GetId(), regionEight);

    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackTwo);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackThree);
    UNIT_ASSERT_VALUES_EQUAL(hackOne, hackFour);
}

Y_UNIT_TEST_SUITE(TranslocalityTests) {
    Y_UNIT_TEST(TestTranslocalParent) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            // Simple case with no hacks
            for (const auto& case_ : ID_AND_PARENT) {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto parent = TranslocalParentID(case_.ID, tld, *geodb, hack);
                UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);
                UNIT_ASSERT_VALUES_EQUAL(case_.Parent, parent);
            }

            // Here goes tranlocality
            {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto parent = TranslocalParentID(REPUBLIC_OF_CRIMEA, tld, *geodb, hack);
                if (YST_RU == tld || YST_BY == tld || YST_KZ == tld) {
                    UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);
                    UNIT_ASSERT_VALUES_EQUAL(parent, CRIMEAN_FEDERAL_DISTRICT);
                } else {
                    UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);
                    UNIT_ASSERT_VALUES_EQUAL(parent, UKRAINE);
                }
            }
        }
    }

    Y_UNIT_TEST(TestTranlocalParentConsistency) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto id : ALL_KNOWN_IDS) {
                CheckTranslocalParentConsistency(id, tld, *geodb);
            }
        }
    }

    Y_UNIT_TEST(TestTranlocalParentConsistencyForAll) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto& value : *geodb) {
                const auto id = value.first;
                CheckTranslocalParentConsistency(id, tld, *geodb);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalParentWithoutHackIsParent) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto id : ALL_KNOWN_IDS) {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto parent = TranslocalParentID(id, tld, *geodb, hack);
                if (NGeoDB::NO_HACK != hack) {
                    continue;
                }

                const auto expectedParent = geodb->Find(id)->GetParentId();
                UNIT_ASSERT_VALUES_EQUAL(expectedParent, parent);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalParentWithoutHackIsParentForAll) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto& value : *geodb) {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto parent = TranslocalParentID(value.first, tld, *geodb, hack);
                if (NGeoDB::NO_HACK != hack) {
                    continue;
                }

                const auto expectedParent = value.second.GetParentId();
                UNIT_ASSERT_VALUES_EQUAL(expectedParent, parent);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalCountry) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            // Simple case with no hacks
            for (const auto& case_ : ID_AND_COUNTRY) {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto country = TranslocalCountryID(case_.ID, tld, *geodb, hack);
                UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);
                UNIT_ASSERT_VALUES_EQUAL(case_.Country, country);
            }

            // Here goes tranlocality
            {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto country = TranslocalCountryID(SEVASTOPOL, tld, *geodb, hack);
                if (YST_RU == tld || YST_BY == tld || YST_KZ == tld) {
                    UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);
                    UNIT_ASSERT_VALUES_EQUAL(country, RUSSIA);
                } else {
                    UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);
                    UNIT_ASSERT_VALUES_EQUAL(country, UKRAINE);
                }
            }
        }
    }

    Y_UNIT_TEST(TestTranlocalCountryConsistency) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto id : ALL_KNOWN_IDS) {
                CheckTranslocalCountryConsistency(id, tld, *geodb);
           }
        }
    }

    Y_UNIT_TEST(TestTranlocalCountryConsistencyForAll) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto& value : *geodb) {
                const auto id = value.first;
                CheckTranslocalCountryConsistency(id, tld, *geodb);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalCountryOfCountryIsCountry) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto id : COUNTRIES) {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto country = TranslocalCountryID(id, tld, *geodb, hack);
                UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);
                UNIT_ASSERT_VALUES_EQUAL(id, country);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalCountryOfCountryIsCountryForAll) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto& value : *geodb) {
                if (NGeoDB::COUNTRY != value.second.GetType()) {
                    continue;
                }

                const auto id = value.first;
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto country = TranslocalCountryID(id, tld, *geodb, hack);
                UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);
                UNIT_ASSERT_VALUES_EQUAL(id, country);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalCountryWithoutHackIsCountry) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto id : ALL_KNOWN_IDS) {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto parent = TranslocalCountryID(id, tld, *geodb, hack);
                if (NGeoDB::NO_HACK != hack) {
                    continue;
                }

                const auto expectedCountry = geodb->Find(id)->GetCountryId();
                UNIT_ASSERT_VALUES_EQUAL(expectedCountry, parent);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalCountryWithoutHackIsCountryForAll) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto& value : *geodb) {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto parent = TranslocalCountryID(value.first, tld, *geodb, hack);
                if (NGeoDB::NO_HACK != hack) {
                    continue;
                }

                const auto expectedCountry = value.second.GetCountryId();
                UNIT_ASSERT_VALUES_EQUAL(expectedCountry, parent);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalCountryOfRegionAboveCountryIsEndCateg) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        auto ids = THashSet<TCateg>{};
        for (const auto country : COUNTRIES) {
            const auto countryWithData = geodb->Find(country);
            for (const auto region : countryWithData->GetPath()) {
                ids.insert(region);
            }
        }

        for (const auto tld : RELEVANT_TLDS) {
            for (const auto id : ids) {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto country = TranslocalCountryID(id, tld, *geodb, hack);
                UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);
                UNIT_ASSERT_VALUES_EQUAL(END_CATEG, country);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalCountryOfRegionAboveCountryIsEndCategForAll) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        auto ids = THashSet<TCateg>{};
        for (const auto& value : *geodb) {
            if (NGeoDB::COUNTRY != value.second.GetType()) {
                continue;
            }

            for (const auto region : value.second.GetPath()) {
                ids.insert(region);
            }
        }

        for (const auto tld : RELEVANT_TLDS) {
            for (const auto id : ids) {
                auto hack = NGeoDB::ETranslocalityHack{};
                const auto country = TranslocalCountryID(id, tld, *geodb, hack);
                UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);
                UNIT_ASSERT_VALUES_EQUAL(END_CATEG, country);
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalIsInSimpleCases) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto idAndParent : ID_AND_PARENT) {
                auto hack = NGeoDB::ETranslocalityHack{};
                UNIT_ASSERT(TranslocalIsIn(idAndParent.ID, idAndParent.Parent, tld, *geodb, hack));
                UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

                CheckTranslocalIsInConsistency(idAndParent.ID, idAndParent.Parent, tld, *geodb);
            }

            for (const auto idAndCountry : ID_AND_COUNTRY) {
                auto hack = NGeoDB::ETranslocalityHack{};
                UNIT_ASSERT(TranslocalIsIn(idAndCountry.ID, idAndCountry.Country, tld, *geodb, hack));
                UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

                CheckTranslocalIsInConsistency(idAndCountry.ID, idAndCountry.Country, tld, *geodb);
            }
        }

        for (const auto tld : RELEVANT_TLDS) {
            for (auto index = size_t{}; index < ID_AND_PARENT.size(); ++index) {
                for (auto jindex = size_t{}; jindex < ID_AND_PARENT.size(); ++jindex) {
                    if (index == jindex) {
                        continue;
                    }

                    const auto child = ID_AND_PARENT[index].ID;
                    const auto parent = ID_AND_PARENT[jindex].Parent;
                    auto hack = NGeoDB::ETranslocalityHack{};
                    UNIT_ASSERT(!TranslocalIsIn(child, parent, tld, *geodb, hack));
                    UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

                    CheckTranslocalIsInConsistency(child, parent, tld, *geodb);
                }
            }
        }

        for (const auto tld : RELEVANT_TLDS) {
            for (auto index = size_t{}; index < ID_AND_COUNTRY.size(); ++index) {
                for (auto jindex = size_t{}; jindex < ID_AND_COUNTRY.size(); ++jindex) {
                    if (index == jindex) {
                        continue;
                    }

                    const auto child = ID_AND_COUNTRY[index].ID;
                    const auto parent = ID_AND_COUNTRY[jindex].Country;
                    auto hack = NGeoDB::ETranslocalityHack{};
                    UNIT_ASSERT(!TranslocalIsIn(child, parent, tld, *geodb, hack));
                    UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

                    CheckTranslocalIsInConsistency(child, parent, tld, *geodb);
                }
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalIsInForCrimea) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(TranslocalIsIn(SEVASTOPOL, RUSSIA, YST_RU, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, RUSSIA, YST_RU, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(TranslocalIsIn(SEVASTOPOL, RUSSIA, YST_BY, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, RUSSIA, YST_BY, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(TranslocalIsIn(SEVASTOPOL, RUSSIA, YST_KZ, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, RUSSIA, YST_KZ, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(!TranslocalIsIn(SEVASTOPOL, RUSSIA, YST_UA, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, RUSSIA, YST_UA, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(!TranslocalIsIn(SEVASTOPOL, RUSSIA, YST_COM, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, RUSSIA, YST_COM, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(!TranslocalIsIn(SEVASTOPOL, RUSSIA, YST_TR, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, RUSSIA, YST_TR, *geodb);
        }

        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(!TranslocalIsIn(SEVASTOPOL, UKRAINE, YST_RU, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, UKRAINE, YST_RU, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(!TranslocalIsIn(SEVASTOPOL, UKRAINE, YST_BY, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, UKRAINE, YST_BY, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(!TranslocalIsIn(SEVASTOPOL, UKRAINE, YST_KZ, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, UKRAINE, YST_KZ, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(TranslocalIsIn(SEVASTOPOL, UKRAINE, YST_UA, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, UKRAINE, YST_UA, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(TranslocalIsIn(SEVASTOPOL, UKRAINE, YST_COM, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, UKRAINE, YST_COM, *geodb);
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT(TranslocalIsIn(SEVASTOPOL, UKRAINE, YST_TR, *geodb, hack));
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);

            CheckTranslocalIsInConsistency(SEVASTOPOL, UKRAINE, YST_TR, *geodb);
        }
    }

    Y_UNIT_TEST(TestTranslocalParentExcludingTypes) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT_VALUES_EQUAL(
                TranslocalParentIDExcludingTypes(
                    KHAMOVNIKI_DISTRICT, YST_RU, *geodb, {NGeoDB::CITY_AREA}, hack
                ),
                MOSCOW
            );
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalParentExcludingTypesConsistency(KHAMOVNIKI_DISTRICT, YST_RU, *geodb, {NGeoDB::CITY_AREA});
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT_VALUES_EQUAL(
                TranslocalParentIDExcludingTypes(SEVASTOPOL, YST_RU, *geodb, {NGeoDB::CONSTITUENT_ENTITY, NGeoDB::FEDERAL_DISTRICT}, hack),
                RUSSIA
            );
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalParentExcludingTypesConsistency(SEVASTOPOL, YST_RU, *geodb, {NGeoDB::CONSTITUENT_ENTITY, NGeoDB::FEDERAL_DISTRICT});
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT_VALUES_EQUAL(
                TranslocalParentIDExcludingTypes(SEVASTOPOL, YST_UA, *geodb, {NGeoDB::CONSTITUENT_ENTITY, NGeoDB::FEDERAL_DISTRICT}, hack),
                UKRAINE
            );
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);

            CheckTranslocalParentExcludingTypesConsistency(SEVASTOPOL, YST_UA, *geodb, {NGeoDB::CONSTITUENT_ENTITY, NGeoDB::FEDERAL_DISTRICT});
        }
    }

    Y_UNIT_TEST(TestTranslocalReduceExcludingTypes) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT_VALUES_EQUAL(
                TranslocalReduceIDExcludingTypes(
                    KHAMOVNIKI_DISTRICT, YST_RU, *geodb, {NGeoDB::CITY_AREA, NGeoDB::CITY_SUBAREA}, hack
                ),
                MOSCOW
            );
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalReduceExcludingTypesConsistency(KHAMOVNIKI_DISTRICT, YST_RU, *geodb, {NGeoDB::CITY_AREA, NGeoDB::CITY_SUBAREA});
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT_VALUES_EQUAL(
                TranslocalReduceIDExcludingTypes(SEVASTOPOL, YST_RU, *geodb, {NGeoDB::CONSTITUENT_ENTITY, NGeoDB::FEDERAL_DISTRICT, NGeoDB::CITY}, hack),
                RUSSIA
            );
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

            CheckTranslocalReduceExcludingTypesConsistency(SEVASTOPOL, YST_RU, *geodb, {NGeoDB::CONSTITUENT_ENTITY, NGeoDB::FEDERAL_DISTRICT});
        }
        {
            auto hack = NGeoDB::ETranslocalityHack{};
            UNIT_ASSERT_VALUES_EQUAL(
                TranslocalReduceIDExcludingTypes(SEVASTOPOL, YST_UA, *geodb, {NGeoDB::CONSTITUENT_ENTITY, NGeoDB::FEDERAL_DISTRICT, NGeoDB::CITY}, hack),
                UKRAINE
            );
            UNIT_ASSERT_VALUES_EQUAL(NGeoDB::CRIMEA_HACK, hack);

            CheckTranslocalReduceExcludingTypesConsistency(SEVASTOPOL, YST_UA, *geodb, {NGeoDB::CONSTITUENT_ENTITY, NGeoDB::FEDERAL_DISTRICT});
        }
    }

    Y_UNIT_TEST(TestTranslocalParentExcludingTypesForAll) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto& value : *geodb) {
                const auto region = value.first;
                auto hackOne = NGeoDB::ETranslocalityHack{};
                auto hackTwo = NGeoDB::ETranslocalityHack{};
                UNIT_ASSERT_VALUES_EQUAL(
                    TranslocalParentIDExcludingTypes(region, tld, *geodb, {}, hackOne),
                    TranslocalParentID(region, tld, *geodb, hackTwo)
                );
                UNIT_ASSERT_VALUES_EQUAL(hackOne, hackTwo);

                CheckTranslocalParentExcludingTypesConsistency(region, tld, *geodb, {});
            }
        }
    }

    Y_UNIT_TEST(TestTranslocalReduceExcludingTypesForAll) {
        const auto geodb = NGeoDB::TGeoKeeper::LoadToHolder(GEODB_PATH);
        for (const auto tld : RELEVANT_TLDS) {
            for (const auto& value : *geodb) {
                const auto region = value.first;
                auto hack = NGeoDB::ETranslocalityHack{};
                UNIT_ASSERT_VALUES_EQUAL(
                    TranslocalReduceIDExcludingTypes(region, tld, *geodb, {}, hack),
                    region
                );
                UNIT_ASSERT_VALUES_EQUAL(NGeoDB::NO_HACK, hack);

                CheckTranslocalReduceExcludingTypesConsistency(region, tld, *geodb, {});
            }
        }
    }
}
