#include <library/cpp/langs/langs.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/charset/wide.h>
#include <util/generic/string.h>

#include "pluralize.h"

using NInfl::Pluralize;
using NInfl::Singularize;

static void TestPluralizeRussianWord(const TString& utf8Word, const TString& utf8Word2, const TString& utf8Word5) {
    auto word = UTF8ToWide(utf8Word);
    auto word2 = UTF8ToWide(utf8Word2);
    auto word5 = UTF8ToWide(utf8Word5);
    for (unsigned int n : {1, 21, 1001})
        UNIT_ASSERT_EQUAL_C(Pluralize(word, n, LANG_RUS), word, utf8Word);

    for (unsigned int n : {2, 3, 4, 33, 44})
        UNIT_ASSERT_EQUAL_C(Pluralize(word, n, LANG_RUS), word2, utf8Word);

    for (unsigned int n : {0, 5, 6, 7, 8, 9, 10, 11, 12, 17, 20, 26, 38, 50, 100, 111, 256})
        UNIT_ASSERT_EQUAL_C(Pluralize(word, n, LANG_RUS), word5, utf8Word);
}

static void TestSingularizeRussianWord(const TString& utf8Word, const TString& utf8Word2, const TString& utf8Word5) {
    auto word = UTF8ToWide(utf8Word);
    auto word2 = UTF8ToWide(utf8Word2);
    auto word5 = UTF8ToWide(utf8Word5);

    for (unsigned int n : {1, 21, 1001})
        UNIT_ASSERT_EQUAL_C(Singularize(word, n, LANG_RUS), word, utf8Word);

    for (unsigned int n : {2, 3, 4, 33, 44})
        UNIT_ASSERT_EQUAL_C(Singularize(word2, n, LANG_RUS), word, utf8Word);

    for (unsigned int n : {0, 5, 6, 7, 8, 9, 10, 11, 12, 17, 20, 26, 38, 50, 100, 111, 256})
        UNIT_ASSERT_EQUAL_C(Singularize(word5, n, LANG_RUS), word, utf8Word);
}

static TVector<TVector<unsigned int>> numbers = {
    {0},
    {5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 17, 19, 20, 77, 128},
    {1, 21, 31, 51, 71, 91, 101, 121, 131, 241, 391, 401, 431, 441, 491},
    {2, 3, 4, 22, 54, 73, 92, 93, 94, 102, 103, 104, 202, 444},
};

static TVector<TVector<unsigned int>> numbersAnimal = {
    {0},
    {5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 17, 19, 20, 77, 128},
    {1, 21, 31, 51, 71, 91, 101, 121, 131, 241, 391, 401, 431, 441, 491},
    {2, 3, 4},
    {22, 54, 73, 92, 93, 94, 102, 103, 104, 202, 444},
};

static void TestPluralizeRussianWordWithCaseInanimated(
        TVector<TString> wordSet,
        const TString& utf8WordSingle,
        size_t index) {
    auto wordSingle = UTF8ToWide(utf8WordSingle);
    auto wordNom = UTF8ToWide(wordSet[0]);
    auto wordGen = UTF8ToWide(wordSet[1]);
    auto wordDat = UTF8ToWide(wordSet[2]);
    auto wordAcc = UTF8ToWide(wordSet[3]);
    auto wordIns = UTF8ToWide(wordSet[4]);
    auto wordAbl = UTF8ToWide(wordSet[5]);
    for (unsigned int n : numbers[index]) {
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gNominative, LANG_RUS), wordNom, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gGenitive, LANG_RUS), wordGen, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gDative, LANG_RUS), wordDat, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gAccusative, LANG_RUS), wordAcc, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gInstrumental, LANG_RUS), wordIns, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gAblative, LANG_RUS), wordAbl, utf8WordSingle);
    }
}

static void TestPluralizeRussianWordWithCaseAnimated(
        TVector<TString> wordSet,
        const TString& utf8WordSingle,
        size_t index) {
    auto wordSingle = UTF8ToWide(utf8WordSingle);
    auto wordNom = UTF8ToWide(wordSet[0]);
    auto wordGen = UTF8ToWide(wordSet[1]);
    auto wordDat = UTF8ToWide(wordSet[2]);
    auto wordAcc = UTF8ToWide(wordSet[3]);
    auto wordIns = UTF8ToWide(wordSet[4]);
    auto wordAbl = UTF8ToWide(wordSet[5]);

    for (unsigned int n : numbersAnimal[index]) {
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gNominative, LANG_RUS, true), wordNom, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gGenitive, LANG_RUS, true), wordGen, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gDative, LANG_RUS, true), wordDat, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gAccusative, LANG_RUS, true), wordAcc, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gInstrumental, LANG_RUS, true), wordIns, utf8WordSingle);
        UNIT_ASSERT_EQUAL_C(Pluralize(wordSingle, n, gAblative, LANG_RUS, true), wordAbl, utf8WordSingle);
    }
}

static const char* Words[][3] = {
    {"попугай", "попугая", "попугаев"},
    {"ночь", "ночи", "ночей"},
    {"корова", "коровы", "коров"},
    {"рубль", "рубля", "рублей"},
    {"ребенок", "ребенка", "детей"},
    {"веселый чиж", "веселых чижа", "веселых чижей"},
    {"мгновение весны", "мгновения весны", "мгновений весны"},
    {"бакинский комиссар", "бакинских комиссара", "бакинских комиссаров"},
    {"московский комсомолец", "московских комсомольца", "московских комсомольцев"},
    {"лежачий полицейский", "лежачих полицейских", "лежачих полицейских"},
    {"танкист - веселый друг", "танкиста - веселых друга", "танкистов - веселых друзей"},
    //{"стакан муки", "стакана муки", "стаканов муки"},
    //{"чайная ложка сахара", "чайные ложки сахара", "чайных ложек сахара"},
    //{"оттенок серого", "оттенка серого", "оттенков серого"},
};

static TVector<TVector<TVector<TString>>> WordsInanimated = {
    {
        {"дней", "дней", "дней", "дней", "дней", "дней"},
        {"дней", "дней", "дням", "дней", "днями", "днях"},
        {"день", "дня", "дню", "день", "днем", "дне"},
        {"дня", "дней", "дням", "дня", "днями", "днях"},
    },
    {
        {"желтых автобусов", "желтых автобусов", "желтых автобусов", "желтых автобусов", "желтых автобусов", "желтых автобусов"}, //= 0
        {"желтых автобусов", "желтых автобусов", "желтым автобусам", "желтых автобусов", "желтыми автобусами", "желтых автобусах"}, // mod10 : 0, 5..9; mod100 : 11..14
        {"желтый автобус", "желтого автобуса", "желтому автобусу", "желтый автобус", "желтым автобусом", "желтом автобусе"}, // mod10 : 1
        {"желтых автобуса", "желтых автобусов", "желтым автобусам", "желтых автобуса", "желтыми автобусами", "желтых автобусах"}, // mod10: 2..4
    },
    {
        {"белых стульев", "белых стульев", "белых стульев", "белых стульев", "белых стульев", "белых стульев"},
        {"белых стульев", "белых стульев", "белым стульям", "белых стульев", "белыми стульями", "белых стульях"},
        {"белый стул", "белого стула", "белому стулу", "белый стул", "белым стулом", "белом стуле"},
        {"белых стула", "белых стульев", "белым стульям", "белых стула", "белыми стульями", "белых стульях"},
    },
    {
        {"ярких ламп", "ярких ламп", "ярких ламп", "ярких ламп", "ярких ламп", "ярких ламп"},
        {"ярких ламп", "ярких ламп", "ярким лампам", "ярких ламп", "яркими лампами", "ярких лампах"},
        {"яркая лампа", "яркой лампы", "яркой лампе", "яркую лампу", "яркой лампой", "яркой лампе"},
        {"яркие лампы", "ярких ламп", "ярким лампам", "яркие лампы", "яркими лампами", "ярких лампах"},
    },
    {
        {"больших вселенных", "больших вселенных", "больших вселенных", "больших вселенных", "больших вселенных", "больших вселенных"},
        {"больших вселенных", "больших вселенных", "большим вселенным", "больших вселенных", "большими вселенными", "больших вселенных"},
        {"большая вселенная", "большой вселенной", "большой вселенной", "большую вселенную", "большой вселенной", "большой вселенной"},
        {"большие вселенные", "больших вселенных", "большим вселенным", "большие вселенные", "большими вселенными", "больших вселенных"},
    },
    {
        {"лежачих полицейских", "лежачих полицейских", "лежачих полицейских", "лежачих полицейских", "лежачих полицейских", "лежачих полицейских"},
        {"лежачих полицейских", "лежачих полицейских", "лежачим полицейским", "лежачих полицейских", "лежачими полицейскими", "лежачих полицейских"},
        {"лежачий полицейский", "лежачего полицейского", "лежачему полицейскому", "лежачий полицейский", "лежачим полицейским", "лежачем полицейском"},
        {"лежачих полицейских", "лежачих полицейских", "лежачим полицейским", "лежачих полицейских", "лежачими полицейскими", "лежачих полицейских"},
    },
    // {
    //     {"городов-героев", "городов-героев", "городов-героев", "городов-героев", "городов-героев", "городов-героев"},
    //     {"городов-героев", "городов-героев", "городам-героям", "городов-героев", "городами-героями", "городах-героях"},
    //     {"город-герой", "города-героя", "городу-герою", "город-герой", "городом-героем", "городе-герое"},
    //     {"города-героя", "городов-героев", "городам-героям", "города-героя", "городами-героями", "городах-героях"},
    // },
};

static TVector<TString> WordsInanimatedSingNom = {
    "день",
    "желтый автобус",
    "белый стул",
    "яркая лампа",
    "большая вселенная",
    "лежачий полицейский",
    // "город-герой",
};

static TVector<TVector<TVector<TString>>> WordsAnimated = {
    {
        {"котов", "котов", "котов", "котов", "котов", "котов"},
        {"котов", "котов", "котам", "котов", "котами", "котах"},
        {"кот", "кота", "коту", "кота", "котом", "коте"},
        {"кота", "котов", "котам", "котов", "котами", "котах"},
        {"кота", "котов", "котам", "кота", "котами", "котах"},
    },
    {
        {"веселых детей", "веселых детей", "веселых детей", "веселых детей", "веселых детей", "веселых детей"}, //= 0
        {"веселых детей", "веселых детей", "веселым детям", "веселых детей", "веселыми детьми", "веселых детях"}, // mod10 : 0, 5..9; mod100 : 11..14
        {"веселый ребенок", "веселого ребенка", "веселому ребенку", "веселого ребенка", "веселым ребенком", "веселом ребенке"}, //1,21,31,41,...
        {"веселых ребенка", "веселых детей", "веселым детям", "веселых детей", "веселыми детьми", "веселых детях"}, //  2..4
        {"веселых ребенка", "веселых детей", "веселым детям", "веселых ребенка", "веселыми детьми", "веселых детях"}, //mod100 : 22..24, 32..34, ...

    },
    {
        {"добрых псов", "добрых псов", "добрых псов", "добрых псов", "добрых псов", "добрых псов"},
        {"добрых псов", "добрых псов", "добрым псам", "добрых псов", "добрыми псами", "добрых псах"},
        {"добрый пёс", "доброго пса", "доброму псу", "доброго пса", "добрым псом", "добром псе"},
        {"добрых пса", "добрых псов", "добрым псам", "добрых псов", "добрыми псами", "добрых псах"},
        {"добрых пса", "добрых псов", "добрым псам", "добрых пса", "добрыми псами", "добрых псах"},
    },
    {
        {"певчих птиц", "певчих птиц", "певчих птиц", "певчих птиц", "певчих птиц", "певчих птиц"},
        {"певчих птиц", "певчих птиц", "певчим птицам", "певчих птиц", "певчими птицами", "певчих птицах"},
        {"певчая птица", "певчей птицы", "певчей птице", "певчую птицу", "певчей птицей", "певчей птице"},
        {"певчие птицы", "певчих птиц", "певчим птицам", "певчих птиц", "певчими птицами", "певчих птицах"},
        {"певчие птицы", "певчих птиц", "певчим птицам", "певчие птицы", "певчими птицами", "певчих птицах"},
    },
    {
        {"красивых горничных", "красивых горничных", "красивых горничных", "красивых горничных", "красивых горничных", "красивых горничных"},
        {"красивых горничных", "красивых горничных", "красивым горничным", "красивых горничных", "красивыми горничными", "красивых горничных"},
        {"красивая горничная", "красивой горничной", "красивой горничной", "красивую горничную", "красивой горничной", "красивой горничной"},
        {"красивые горничные", "красивых горничных", "красивым горничным", "красивых горничных", "красивыми горничными", "красивых горничных"},
        {"красивые горничные", "красивых горничных", "красивым горничным", "красивые горничные", "красивыми горничными", "красивых горничных"},
    },
    {
        {"веб-разработчиков", "веб-разработчиков", "веб-разработчиков", "веб-разработчиков", "веб-разработчиков", "веб-разработчиков"},
        {"веб-разработчиков", "веб-разработчиков", "веб-разработчикам", "веб-разработчиков", "веб-разработчиками", "веб-разработчиках"},
        {"веб-разработчик", "веб-разработчика", "веб-разработчику", "веб-разработчика", "веб-разработчиком", "веб-разработчике"},
        {"веб-разработчика", "веб-разработчиков", "веб-разработчикам", "веб-разработчиков", "веб-разработчиками", "веб-разработчиках"},
        {"веб-разработчика", "веб-разработчиков", "веб-разработчикам", "веб-разработчика", "веб-разработчиками", "веб-разработчиках"},
    },
    {
        {"слесарей-сантехников", "слесарей-сантехников", "слесарей-сантехников", "слесарей-сантехников", "слесарей-сантехников", "слесарей-сантехников"},
        {"слесарей-сантехников", "слесарей-сантехников", "слесарям-сантехникам", "слесарей-сантехников", "слесарями-сантехниками", "слесарях-сантехниках"},
        {"слесарь-сантехник", "слесаря-сантехника", "слесарю-сантехнику", "слесаря-сантехника", "слесарем-сантехником", "слесаре-сантехнике"},
        {"слесаря-сантехника", "слесарей-сантехников", "слесарям-сантехникам", "слесарей-сантехников", "слесарями-сантехниками", "слесарях-сантехниках"},
        {"слесаря-сантехника", "слесарей-сантехников", "слесарям-сантехникам", "слесаря-сантехника", "слесарями-сантехниками", "слесарях-сантехниках"},
    }
};

static TVector<TString> WordsAnimatedSingNom = {
    "кот",
    "веселый ребенок",
    "добрый пёс",
    "певчая птица",
    "красивая горничная",
    "веб-разработчик",
    "слесарь-сантехник",
};

Y_UNIT_TEST_SUITE(TPluralizeTest) {

Y_UNIT_TEST(TestPluralizeRussianWords) {
    for (const auto& wordSet : Words) {
        TestPluralizeRussianWord(wordSet[0], wordSet[1], wordSet[2]);
    }
}

Y_UNIT_TEST(TestSingularizeRussianWords) {
    for (const auto& wordSet : Words) {
        TestSingularizeRussianWord(wordSet[0], wordSet[1], wordSet[2]);
    }
}

Y_UNIT_TEST(TestPluralizeRussianWordsWithCaseInanimated) {
    size_t j = 0;
    for (const auto& wordSet : WordsInanimated) {
        size_t i = 0;
        for (const auto& wordCase : wordSet) {
            TestPluralizeRussianWordWithCaseInanimated(wordCase, WordsInanimatedSingNom[j], i);
            i++;
        }
        j++;
    }
}

Y_UNIT_TEST(TestPluralizeRussianWordsWithCaseAnimated) {
    size_t j = 0;
    for (const auto& wordSet : WordsAnimated) {
        size_t i = 0;
        for (const auto& wordCase : wordSet) {
            TestPluralizeRussianWordWithCaseAnimated(wordCase, WordsAnimatedSingNom[j], i);
            i++;
        }
        j++;
    }
}

}

