# -*- coding: utf-8 -*-


"""Library for generating alphabets and translations C++ code.

The library provides several functions for manipulations  with mappings
(dicts) and classes for code generation from lists and mappings.  The
main code generation class is CodeBuilder, which is the parent class
for all other classes.  In order to create a piece of code with
specific data, you need to create an object of the corresponding class
and pass it to the generate_code() function.  Each code building class
generates some code and defines a list of necessary file headers, which
are managed automatically by the generate_code() function.

"""


import logging
import itertools


logging.basicConfig(level=logging.ERROR)


def get_symbols_range(begin, end):
    return [unichr(i) for i in xrange(begin, end)]


def compose(*mappers):

    """Compose a sequence of mappings."""

    keys = set()
    for mapper in mappers:
        keys = keys.union(set(mapper.keys()))

    result = {}
    for symbol in keys:
        result_symbol = symbol
        for mapper in mappers:
            result_symbol = mapper.get(result_symbol, result_symbol)
        result[symbol] = result_symbol

    return result


def reverse_mapping(mapping):
    """Reverse the keys and values in the mapping.

    Considering values are unique.

    """
    result = dict((value, key) for key, value in mapping.items())
    assert len(result) == len(mapping), "Values must be unique."
    return result


def merge(*mappers):
    """Merge a sequence of mappings.

    Considering keys do not intersect.

    """
    result = {}
    for mapper in mappers:
        result.update(mapper)
    return result


def _remove_nonchanging(mapping):
    """Remove the pairs where key matches value from the mapping.

    This is necessary to reduce the resulting generated code size.

    """
    return dict((key, value) for key, value in mapping.items() if key != value)


def _mapping_to_lists(mapping):

    """Generate lists with 'from' and 'to' values by a mapping."""

    list_from = []
    list_to = []
    for key, value in mapping.items():
        list_from.append(key)
        list_to.append(value)

    return list_from, list_to


class CodeBuilder(object):

    """The base class for all code generation classes.

    All inheritors have to override the main code generation functions:
        generate_code - this function returns the code generated by the
            object (except 'headers', see get_cpp_headers)
        get_cpp_headers - this function returns the list of lines that
            must be present in the header so that the generated code
            can work correctly.

    """

    def generate_code(self):
        return ''

    def __str__(self):
        return 'CodeBuilder()'

    @staticmethod
    def get_cpp_headers():
        return ()


class TRawCode(CodeBuilder):
    """to insert code directly"""
    def __init__(self, code):
        self.code = code

    def generate_code(self):
        return self.code


class ConstantCode(CodeBuilder):

    """Generate code for C++ constants."""

    def __init__(self, type_, name, value):
        self.type_ = type_
        self.name = name
        self.value = value

    __CODE_TEMPLATE = "constexpr {type_} {name} = {value};\n"

    def generate_code(self):
        return self.__CODE_TEMPLATE.format(
            type_=self.type_, name=self.name, value=self.value)

    def __str__(self):
        return "ConstantCode(name={name})".format(name=self.name)


class StaticArrayCode(CodeBuilder):

    """Generate code for static arrays with given elements.

    The elements may be generic, but it is necessary to define a
    function to convert element to their string representation.

    """

    def __init__(self, header, elements, element_to_code=lambda x: x):
        self.header = header
        elements_codes = map(element_to_code, elements)
        self.elements_code = ', '.join(elements_codes + ['0'])

    __CODE_TEMPLATE = "{header}[] = {{ {elements} }};\n"

    def generate_code(self):
        return self.__CODE_TEMPLATE.format(header=self.header,
                                           elements=self.elements_code)

    def __str__(self):
        return "StaticArrayCode(header={header})".format(header=self.header)


def str_to_hex(symbol):
    """Get string hex representation of the symbol."""
    return hex(ord(symbol))


class WCharStaticArrayCode(StaticArrayCode):

    """Generate code for static array of wchar16 elements.

    The elements of the array must be unicode symbols, and they are
    converted to the corresponding hex string representations.

    """

    def __init__(self, name, elements):
        header = 'constexpr wchar16 ' + name
        assert all(isinstance(el, unicode) and len(el) == 1 for
                   el in elements
                   ), "All array elements must be unicode strings of length 1."
        StaticArrayCode.__init__(self, header, elements, str_to_hex)


class WCharStaticArrayPackCode(CodeBuilder):

    """Generate code for a pack of static wchar array.

    The generated code contains data for a list of arrays with
    different names.  The names can be retrieved from 'names'
    attribute.  The names_prefix constructor argument must be chosen
    so that no other names in code have this prefix.

    """

    def __init__(self, names_prefix, lists):
        self.names_prefix = names_prefix
        self.lists = lists
        self.names = [names_prefix + str(index) for
                      index in xrange(len(self.lists))]

    def generate_code(self):
        result_parts = []
        for name, elements in zip(self.names, self.lists):
            code = WCharStaticArrayCode(name, elements)
            result_parts.append(code.generate_code())
        return ''.join(result_parts)

    def __str__(self):
        return "WCharStaticArrayPackCode(names_prefix={names_prefix})".format(
            names_prefix=self.names_prefix)


class TTrClassCode(CodeBuilder):

    """Generate code for a TTr class inheritor.

    The class depends on three tables: 'simple' mapping table (one
    symbol to one symbol), 'advanced' mapping table (one symbol to
    several symbols) and 'kill' symbols (symbols that are removed
    completely).  First two are passed a single dict which must have
    unicode symbols as keys and unicode non-empty strings as values
    (they are automatically sorted to 'simple' and 'advanced') and a
    list of symbols to remove.

    The generated code contains the class and all necessary static
    arrays declarations.

    """

    def __init__(self, name, replace_mapping, remove_symbols=(), add_declaration=True):
        self.name = name
        self.constants_prefix = self.name.upper() + '_AUX_'
        self.remove = remove_symbols
        self.add_declaration = add_declaration

        # sort symbols to 'simple' and 'advanced'
        self.replace_one_symbol = {}
        self.replace_several_symbols = {}
        for symbol, replacement in replace_mapping.items():
            assert isinstance(symbol, unicode), \
                "Source symbols must be unicode strings."
            assert len(symbol) == 1, "Source symbols must have length of 1."
            assert isinstance(replacement, unicode), \
                "Replacement symbols must be unicode strings."
            assert len(replacement) > 0, \
                "Replacement symbols must have non-zero length."

            if len(replacement) == 1:
                self.replace_one_symbol[symbol] = replacement
            else:
                self.replace_several_symbols[symbol] = replacement

        self.replace_one_symbol = _remove_nonchanging(self.replace_one_symbol)

    __CONSTRUCTOR_TEMPLATE = """{name}::{name}(): TTr({from_}, {to}, {remove}, {adv_from}, {adv_to}) {{}}\n"""

    __CLASS_CODE_TEMPLATE = (
        "struct {name}: public TTr {{\n"
        "    {name}()\n"
        "        : TTr({from_}, {to}, {remove}, {adv_from}, {adv_to})\n"
        "    {{}}\n"
        "}};\n"
    )

    __EMPTY_ARRAY_NAME = 'NULL'

    def __generate_replace_single(self, result_parts):

        """Generate code for simple mapping.

        The 'result_parts' argument is a list that is modified by this
        function (all necessary code parts are added).

        Return the names of the simple mapping arrays (from and to).

        """

        from_map, to_map = _mapping_to_lists(self.replace_one_symbol)
        from_name = self.constants_prefix + 'FROM'
        to_name = self.constants_prefix + 'TO'

        from_code = WCharStaticArrayCode(from_name, from_map)
        to_code = WCharStaticArrayCode(to_name, to_map)
        result_parts.append(from_code.generate_code())
        result_parts.append(to_code.generate_code())

        return from_name, to_name

    def __generate_remove(self, result_parts):

        """Generate code for removed symbols.

        The 'result_parts' argument is a list that is modified by this
        function (all necessary code parts are added).

        Return the name of the array with removed symbols or 'NULL' if
        there are no symbols to remove.

        """

        if self.remove:
            remove_name = self.constants_prefix + 'KILL'
            remove_code = WCharStaticArrayCode(remove_name, self.remove)
            result_parts.append(remove_code.generate_code())
            return remove_name
        else:
            return self.__EMPTY_ARRAY_NAME

    def __generate_replace_several(self, result_parts):

        """Generate code for removed symbols.

        The 'result_parts' argument is a list that is modified by this
        function (all necessary code parts are added).

        Return the names of the several mapping arrays (from and to).

        """

        if self.replace_several_symbols:
            symbols, replacements = zip(*self.replace_several_symbols.items())
            array_pack = WCharStaticArrayPackCode(
                self.constants_prefix + 'ADV_REPLACE_', replacements)
            result_parts.append(array_pack.generate_code())
            replace_several_from = symbols
            replace_several_to = array_pack.names

            replace_several_from_name = self.constants_prefix + 'ADV_FROM'
            replace_several_to_name = self.constants_prefix + 'ADV_TO'

            replace_several_from_code = WCharStaticArrayCode(
                replace_several_from_name, replace_several_from)
            replace_several_to_code = StaticArrayCode(
                'constexpr const wchar16* ' + replace_several_to_name,
                replace_several_to)

            result_parts.append(replace_several_from_code.generate_code())
            result_parts.append(replace_several_to_code.generate_code())
            return replace_several_from_name, replace_several_to_name
        else:
            return self.__EMPTY_ARRAY_NAME, self.__EMPTY_ARRAY_NAME

    def generate_code(self):
        result_parts = []

        replace_single_from, replace_single_to = (
            self.__generate_replace_single(result_parts))

        remove_name = self.__generate_remove(result_parts)

        replace_several_from_name, replace_several_to_name = (
            self.__generate_replace_several(result_parts))

        template = self.__CLASS_CODE_TEMPLATE if self.add_declaration else self.__CONSTRUCTOR_TEMPLATE
        class_code = template.format(
            name=self.name,
            from_=replace_single_from, to=replace_single_to,
            remove=remove_name,
            adv_from=replace_several_from_name, adv_to=replace_several_to_name)
        result_parts.append(class_code)

        return ''.join(result_parts)

    def __str__(self):
        return "TTrClassCode(name={name})".format(name=self.name)

    @staticmethod
    def get_cpp_headers():
        return (
            '#include <kernel/lemmer/alpha/abc.h>\n',
            'using NLemmer::NDetail::TTr;\n',
        )


class TDiacriticsMapClassCode(CodeBuilder):

    """Generate code for a TDiacriticsMap class inheritor.

    The class depends on a list of diacritics map elements; the
    corresponding static arrays declarations are generated too.

    """

    def __init__(self, name, diacritic_maps):
        self.name = name
        self.diacritic_maps = diacritic_maps

    __CLASS_CODE_TEMPLATE = (
        "class {name}: public TDiacriticsMap {{\n"
        "public:\n"
        "    {name}() {{\n"
        "        {add_maps_code}\n"
        "    }}\n"
        "}};\n"
    )

    __ADD_MAP_CODE_TEMPLATE = "Add({name});\n"

    def generate_code(self):
        result_parts = []

        constants_prefix = self.name.upper() + '_AUX_'

        maps_code = WCharStaticArrayPackCode(constants_prefix,
                                             self.diacritic_maps)
        result_parts.append(maps_code.generate_code())

        add_maps_code_parts = []
        for name in maps_code.names:
            add_code = self.__ADD_MAP_CODE_TEMPLATE.format(name=name)
            add_maps_code_parts.append(add_code)

        class_code = self.__CLASS_CODE_TEMPLATE.format(
            name=self.name, add_maps_code='        '.join(add_maps_code_parts))
        result_parts.append(class_code)

        return ''.join(result_parts)

    def __str__(self):
        return "TDiacriticsMapClassCode(name={name})".format(name=self.name)

    @staticmethod
    def get_cpp_headers():
        return (
            '#include <kernel/lemmer/alpha/abc.h>\n',
            'using NLemmer::TDiacriticsMap;\n',
        )


class TAlphaClassCode(CodeBuilder):
    """Generate code for a TAlpha%LanguageName% class"""
    ARGS = [
        ('required', '{lang}_{arg}_ALPHA'),
        ('normal', '{lang}_{arg}_ALPHA'),
        ('alien', '{lang}_{arg}_ALPHA'),
        ('accidental', '{lang}_STRESS'),
        ('signs', '{lang}_SIGNS')
    ]
    DEFAULT = 'static_cast<const wchar16*>(nullptr)'
    def __init__(self, short_name, long_name, **kwargs):
        self.name = kwargs.get('name', 'TAlpha' + long_name)
        self.args = []
        self.code = []
        for arg, template in self.ARGS:
            if arg not in kwargs:
                self.args.append(self.DEFAULT)
            elif isinstance(kwargs[arg], str):
                self.args.append(kwargs[arg])
            elif isinstance(kwargs[arg], list):
                name = template.format(lang=short_name.upper(), arg=arg.upper())
                self.code.append(WCharStaticArrayCode(name, kwargs[arg]).generate_code())
                self.args.append(name)
            else:
                raise "Argument %s for TAlphaClassCode constructor should be list or string" % (arg, )

    __CLASS_CODE_TEMPLATE = "using {name} = TTemplateAlphabet<{args[0]}, {args[1]}, {args[2]}, {args[3]}, {args[4]}>;\n"

    def generate_code(self):
        return ''.join(self.code + [self.__CLASS_CODE_TEMPLATE.format(**self.__dict__)])

    @staticmethod
    def get_cpp_headers():
        return ('#include <kernel/lemmer/alpha/abc.h>\n', 'using NLemmer::TTemplateAlphabet;\n', )


class TRegisterCode(CodeBuilder):
    LANGUAGES = []

    __REGISTER_CODE_TEMPLATE = '    Register(HugeSingleton<{constructer}>(), {mask});'
    __REGISTER_CODE_TEMPLATE2 = '    Register(HugeSingleton<{constructer}>(), {mask}, HugeSingleton<{constructer2}>());'
    __CLASS_CODE_TEMPLATE = '''void NLemmer::NAlphaDirectory::TAlphaDirectoryImpl::Init() {{
    using namespace NLemmer::NDetail;
    Register(HugeSingleton<TDefaultConverter>(), LANG_UNK);
{registrations}
}}
'''

    @classmethod
    def add_language(cls, short_name, constructer, constructer2, kwargs):
        cls.LANGUAGES.append((short_name, constructer, constructer2, kwargs))

    def generate_code(self):
        usages = []
        registrations = []
        for short_name, constructer, constructer2, kwargs in self.LANGUAGES:
            mask = kwargs.get('mask', 'LANG_%s' % (short_name.upper(), ))
            if constructer2 is not None:
                registrations.append(self.__REGISTER_CODE_TEMPLATE2.format(constructer=constructer, constructer2=constructer2, mask=mask))
            else:
                registrations.append(self.__REGISTER_CODE_TEMPLATE.format(constructer=constructer, mask=mask))
        return self.__CLASS_CODE_TEMPLATE.format(registrations='\n'.join(registrations))

    @staticmethod
    def get_cpp_headers():
        return (
            '#include <kernel/lemmer/alpha/directory.h>\n',
            '#include <util/generic/singleton.h>\n',
            '#include <library/cpp/langs/langs.h>\n',
        )


class TLanguageCode(CodeBuilder):
    DEFAULT_CONVERTERS = {
        "alpha": 'TAlpha{name}',
        "preconverter": 'TDefaultPreConverter',
        "derenyxer": 'TEmptyConverter',
        "converter": 'TApostropheConverter',
        "prelower": 'TEmptyConverter',
        "preupper": 'TEmptyConverter',
        "pretitle": 'TEmptyConverter',
        "diamap": 'TEmptyDiacriticsMap',
    }

    __ALPHA_STRUCT_TYPE_NAME = 'TAlphaStruct{shortname_cap}'
    __ALPHA_STRUCT_TEMPLATE = 'using {constructer} = TAlphaStructConstructer<{alpha}, {preconverter}, {derenyxer}, {converter}, {prelower}, {preupper}, {pretitle}, {diamap}>;\n'

    """Generate all code for specific language"""
    def __init__(self, short_name, long_name, **kwargs):
        self.short_name = short_name
        if long_name == long_name.lower():
            self.long_name = long_name.capitalize()
        else:
            self.long_name = long_name
        self.kwargs = kwargs

    @staticmethod
    def get_cpp_headers():
        return itertools.chain(['#include <kernel/lemmer/alpha/default_converters.h>\n'], *(cl.get_cpp_headers() for cl in [WCharStaticArrayCode, WCharStaticArrayPackCode, TTrClassCode, TAlphaClassCode, TDiacriticsMapClassCode, TRegisterCode]))

    def generate_code(self):
        code = []
        code.append(TAlphaClassCode(self.short_name, self.long_name, **self.kwargs).generate_code())
        if 'constructer' in self.kwargs:
            constructer = self.kwargs['constructer']
        else:
            converters = {c: self.kwargs.get(c, self.DEFAULT_CONVERTERS[c]).format(name=self.long_name) for c in self.DEFAULT_CONVERTERS}
            constructer = self.__ALPHA_STRUCT_TYPE_NAME.format(shortname_cap=self.short_name.capitalize())
            code.append(self.__ALPHA_STRUCT_TEMPLATE.format(constructer=constructer, **converters))

        if 'secondary' in self.kwargs:
            if 'constructer' in self.kwargs['secondary']:
                constructer2 = self.__ALPHA_STRUCT_TYPE_NAME.format(shortname_cap=self.kwargs['secondary']['constructer'].capitalize())
            else:
                code.append(TAlphaClassCode(self.short_name + '2', self.long_name + '2', **self.kwargs['secondary']).generate_code())
                converters = {c: self.kwargs['secondary'].get(c, self.DEFAULT_CONVERTERS[c]).format(name=self.long_name + '2') for c in self.DEFAULT_CONVERTERS}
                constructer2 = self.__ALPHA_STRUCT_TYPE_NAME.format(shortname_cap=self.short_name.capitalize() + '2')
                code.append(self.__ALPHA_STRUCT_TEMPLATE.format(constructer=constructer2, **converters))
        else:
            constructer2 = None

        TRegisterCode.add_language(self.short_name, constructer, constructer2, self.kwargs)
        return ''.join(code)


def _remove_duplicates(sequence):
    """Leave only first occurrence of each element in the sequence.

    The result is a generator.

    """
    elements = set()
    for element in sequence:
        if element not in elements:
            yield element
            elements.add(element)


def generate_code(code_builders, file_):

    """Generate code and write it to the file.

    Arguments:
        code_builders - an iterable of CodeBuilder instances
        file_ - file to write data in

    The '#pragma once' header is added to the code, other headers to
    add are extracted from 'get_cpp_headers' function of the builders.

    """

    header_parts = []
    for builder in code_builders:
        assert isinstance(builder, CodeBuilder), \
            "Builder must inherit from CodeBuilder class"

        logging.debug("Builder " + str(builder) + ", getting C++ header.")
        header_parts.extend(builder.get_cpp_headers())

    for part in _remove_duplicates(header_parts):
        file_.write(part)
    file_.write('\n')

    for builder in code_builders:
        logging.debug("Builder " + str(builder) + ", generating code.")
        code = builder.generate_code()
        file_.write(code + '\n')
