#include "wordform.h"
#include "lemmeraux.h"

using NLemmerAux::TYandexLemmaGetter;

TWordformKit::TWordformKit(const TYandexLemma& lemma, size_t flexIndex)
    : Lemma(&lemma)
    , FlexGrammar(nullptr)
    , Weight(0)
{
    if (flexIndex < lemma.FlexGramNum())
        FlexGrammar = lemma.GetFlexGram()[flexIndex];

    TWtringBuf text(lemma.GetNormalizedForm(), lemma.GetNormalizedFormLength());

    Prefix = text.NextTokAt(lemma.GetPrefLen());
    Postfix = text.SplitOffAt(text.size() - lemma.GetSuffixLength());
    DirectFlexion = text.SplitOffAt(text.size() - lemma.GetFlexLen());
    // the rest goes to StemBase, StemFlexion remains empty
    StemBase = text;
}

void TWordformKit::SetDefaultText(const TYandexLemma& lemma, const TWtringBuf& text) {
    Lemma = &lemma;
    size_t suffixLen = Min<size_t>(lemma.GetSuffixLength(), text.size());
    text.SplitAt(suffixLen, StemBase, Postfix);

    Prefix = TWtringBuf();
    StemFlexion = TWtringBuf();
    InvertedFlexion = TWtringBuf();
    DirectFlexion = TWtringBuf();
}

class TWordformCharIterator {
public:
    TWordformCharIterator(const TWordformKit& kit)
        : Kit(kit)
        , Cur(kit.Prefix.begin())
        , State(0)
    {
        Next();
    }

    void operator++() {
        if (State == 3)
            --Cur;
        else
            ++Cur;
        Next();
    }

    bool Ok() const {
        return State < 6;
    }

    const wchar16* operator->() const {
        return Cur;
    }

    wchar16 operator*() const {
        return *Cur;
    }

private:
    void Next() {
        switch (State) {
            case 0:
                if (Cur < Kit.Prefix.end())
                    return;
                Cur = Kit.StemBase.begin();
                State = 1;
                [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
            case 1:
                if (Cur < Kit.StemBase.end())
                    return;
                Cur = Kit.StemFlexion.begin();
                State = 2;
                [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
            case 2:
                if (Cur < Kit.StemFlexion.end())
                    return;
                if (Kit.InvertedFlexion.empty())
                    goto dirflex;
                Cur = Kit.InvertedFlexion.end() - 1;
                State = 3;
                [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
            case 3:
                if (Cur >= Kit.InvertedFlexion.begin())
                    return;
            dirflex:
                Cur = Kit.DirectFlexion.begin();
                State = 4;
                [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
            case 4:
                if (Cur < Kit.DirectFlexion.end())
                    return;
                Cur = Kit.Postfix.begin();
                State = 5;
                [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
            case 5:
                if (Cur < Kit.Postfix.end())
                    return;
                State = 6;
                [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
            default:
                return;
        }
    }

    const TWordformKit& Kit;
    const wchar16* Cur;
    size_t State;
};

size_t TWordformKit::CommonPrefixLength(const TWtringBuf& str) const {
    size_t common = 0;
    TWordformCharIterator it1(*this);
    TWtringBuf::const_iterator it2(str.begin());
    for (; it1.Ok() && it2 != str.end() && *it1 == *it2; ++it1, ++it2)
        ++common;
    return common;
}

size_t TWordformKit::CommonPrefixLength(const TWordformKit& kit) const {
    size_t common = 0;
    TWordformCharIterator it1(*this);
    TWordformCharIterator it2(kit);
    for (; it1.Ok() && it2.Ok() && *it1 == *it2; ++it1, ++it2)
        ++common;
    return common;
}

bool TWordformKit::HasSameText(const TWordformKit& a) const {
    size_t totalSize = TotalSize();
    return a.TotalSize() == totalSize && CommonPrefixLength(a) == totalSize;
}

// Write as many characters of @str into @buffer as its @bufferSize allows
// and advance @buffer to the end of written data.
static inline void CopyAndAdvance(const TWtringBuf& str, wchar16*& buffer, size_t& bufferSize) {
    size_t written = Min<size_t>(bufferSize, str.size());
    std::char_traits<wchar16>::copy(buffer, str.data(), written);
    buffer += written;
    bufferSize -= written;
}

TWtringBuf TWordformKit::ConstructText(wchar16* buffer, size_t bufferSize) const {
    const wchar16* start = buffer;
    CopyAndAdvance(Prefix, buffer, bufferSize);
    CopyAndAdvance(StemBase, buffer, bufferSize);
    CopyAndAdvance(StemFlexion, buffer, bufferSize);

    const wchar16* flexChar = InvertedFlexion.end();
    while (flexChar > InvertedFlexion.begin())
        CopyAndAdvance(TWtringBuf(--flexChar, 1), buffer, bufferSize);
    CopyAndAdvance(DirectFlexion, buffer, bufferSize);

    CopyAndAdvance(Postfix, buffer, bufferSize);
    return TWtringBuf(start, buffer);
}

void TWordformKit::ConstructText(TUtf16String& text) const {
    size_t totalSize = TotalSize();

    text.clear();
    text.ReserveAndResize(totalSize);
    TWtringBuf constructed = ConstructText(text.begin(), text.size());
    Y_ASSERT(constructed.size() == totalSize);
}

void TWordformKit::ConstructForm(TYandexWordform& form) const {
    if (Lemma == nullptr)
        return;

    NLemmerAux::TYandexWordformSetter setter(form);
    setter.Clear();

    TUtf16String text;
    ConstructText(text);
    setter.SetNormalizedForm(text, Lemma->GetLanguage(), StemBase.size() + StemFlexion.size(), Prefix.size(), Postfix.size());

    setter.SetStemGr(Lemma->GetStemGram());
    if (FlexGrammar != nullptr)
        setter.AddFlexGr(FlexGrammar);
}

TDefaultWordformIterator::TDefaultWordformIterator(const TYandexLemma& lemma)
    : Current(nullptr)
{
    Lemma.SetDefaultText(lemma, TWtringBuf(lemma.GetText(), lemma.GetTextLength()));
    Form.SetDefaultText(lemma, TWtringBuf(lemma.GetNormalizedForm(), lemma.GetNormalizedFormLength()));

    if (lemma.GetTextLength())
        Current = &Lemma;
    else if (lemma.GetNormalizedFormLength())
        Current = &Form;
}
