#pragma once

#include "mn_sse_model.h"

namespace NMatrixnet {

/*! Matrixnet in format optimized for SSE calculations on multiple documents.
 *
 * It is recommend to calculate model value for multiply documents at one call instead
 * of calculating value separate for each document.
 */
class TMnSseDynamic: public TMnSseModel {
public:

    TMnSseDynamic() = default;

    TMnSseDynamic(
            const float* values,      const size_t valuesSize,
            const TFeature* features, const size_t featuresSize,
            const i8* missedValDirs,  const size_t missedValDirsSize,
            const ui32* dInd,         const size_t dIndSize,
            const int* sizeToCount,   const size_t sizeToCountSize,
            const int* data,          const size_t dataSize,
            const double bias,        const double scale);

    /*! Constructor for initializing from dynamic model.
     *
     * Warning! This constructor copy all data.
     */
    TMnSseDynamic(const TMnSseDynamic& matrixnet);
    TMnSseDynamic& operator=(const TMnSseDynamic& matrixnet);

    /*! Constructor for initializing from static data.
     *
     * Warning! This constructor copy all data.
     */
    TMnSseDynamic(const TMnSseStatic& matrixnet);

    /*! Constructor for initializing from static data.
     *
     * Warning! This constructor copy all data.
     */
    TMnSseDynamic(const TMnSseInfo& matrixnet);

    void CopyFrom(
            const float* values,      const size_t valuesSize,
            const TFeature* features, const size_t featuresSize,
            const i8* missedValDirs,  const size_t missedValDirsSize,
            const ui32* dInd,         const size_t dIndSize,
            const int* sizeToCount,   const size_t sizeToCountSize,
            const int* data,          const size_t dataSize,
            const double bias,        const double scale);

    void CopyFrom(const TMnSseStatic& matrixnet);
    void CopyFrom(const TMnSseInfo& matrixnet);

    void Swap(TMnSseDynamic& obj);

    /// Removes all internal data from model so that the model become empty.
    void Clear() override;

    void Load(IInputStream *in) override;

    /*! Split on two models: one with specified factors, other - without.
     *  Both models have the same scale.
     *  First model has zero bias, second has source bias.
     */
    void SplitTreesBySpecifiedFactors(const TSet<ui32>& factorsIndices, TMnSseDynamic &modelWithFactors, TMnSseDynamic &modelWithoutFactors) const;

    /*! Create new model with trees indices in [begin, end).
     *
     * If begin == 0 then bias of the new model is equal to source bias.
     *
     * If begin != 0 then bias of the new model is 0.
     * \warning Model trees order and components may differ from
     *          trees generated by training iterations.
     * \warning RemoveUnusedFeatures is automatically applied to the new model.
    */
    TMnSseDynamic CopyTreeRange(const size_t begin, const size_t end) const;

    /*! Split model into the sum of several model with lesser amount of trees.
     *
     * Bias is stored in the first model. Trees are copied as is then
     * RemoveUnusedFeatures is applied to each part.
    */
    TVector<TMnSseDynamic> SplitTrees(size_t treeCount) const;

    /// Calc sum of two matrixnets
    // alpha * (*this) + (1 - alpha) * mn
    void Add(const TMnSseDynamic &mn, const float alpha = 0.5);

    // alpha * (*this) + beta * mn
    void AddWithCoefficients(const TMnSseDynamic &mn,
                             const float alpha,
                             const float beta);

    // Create dynamic bunle from pairs (model, feature index)
    static TMnSseDynamic CreateDynamicBundle(const TVector<std::pair<TMnSseDynamic, NFactorSlices::TFullFactorIndex>>& models);

    void SetDynamicBundle(TDynamicBundle bundle);

    /*! Add feature to all trees.
     *
     * Add condition factors[featureIdx] > border to all trees.
     * \param featureIdx - condition feature index
     * \param border - condition border
     * \param leftScale - scaling coefficient for values of subtree where condition is false
     * \param rightScale - scaling coefficient for values of subtree where condition is true
     * \warning This operation increases model trees depth. Using trees of depth more than 6 is not recommended.
    */
    void AddFeature(int featureIdx, float border, double leftScale, double rightScale);

private:
    /*! Accessors for single data DataBias & DataScale fields
     *
     * This functions allow easy access to DataScale & DataBias fields
     * calling Matrixnet.Multidata resize for old style models
     * \warning This functions are temporary and dangerous, should be deleted later
     */
    TMultiData::TLeafData& GetSingleMxData() {
        auto& m = std::get<TMultiData>(Matrixnet.Leaves.Data).MultiData;

        Y_ENSURE(m.size() <= 1);
        m.resize(1);
        return m[0];
    }

    const TMultiData::TLeafData& GetSingleMxData() const {
        auto& m = std::get<TMultiData>(Matrixnet.Leaves.Data).MultiData;

        Y_ENSURE(m.size() == 1);
        return m[0];
    }

    TOwnedMultiDataVector& OwnedMultiData() {
        return std::get<TOwnedMultiDataVector>(MultiData);
    }

    const TOwnedMultiDataVector& OwnedMultiData() const {
        return std::get<TOwnedMultiDataVector>(MultiData);
    }
};

typedef TAtomicSharedPtr<const TMnSseDynamic> TMnSseDynamicPtr;

}
