#include <kernel/matrixnet/mn_sse.h>
#include <kernel/factor_slices/factor_borders.h>
#include <kernel/factor_storage/factor_storage.h>
#include <quality/relev_tools/mx_ops/lib/sliced_input_helper.h>
#include <kernel/factor_storage/factors_reader.h>
#include <kernel/factor_storage/factor_storage.h>

#include <library/cpp/archive/yarchive.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/string.h>
#include <util/memory/blob.h>
#include <util/generic/map.h>
#include <util/random/random.h>
#include <util/generic/ymath.h>

using namespace NMatrixnet;
using namespace NFactorSlices;

Y_UNIT_TEST_SUITE(TAnalysisTest) {

namespace {
    extern "C" {
        extern const unsigned char TestModels[];
        extern const ui32 TestModelsSize;
    };

    static const TString Ru("/Ru.info");
    static const TString Fresh("/RuFreshExtRelev.info");
    static const TString IP25("/ip25.info");
    static const TString RandomQuarkSlices("/quark_slices.info");
    static const TString WebL3_2019_07("/mn464170.info");

    template <typename T>
    T ClampedLerp(T a, T b, double x) {
        T raw = a + (b - a) * x;
        return ClampVal<T>(raw, Min(a, b), Max(a, b));
    }

    void SetModel(TMnSseInfo& model, const TString& name) {
        TArchiveReader archive(TBlob::NoCopy(TestModels, TestModelsSize));
        TBlob modelBlob = archive.ObjectBlobByKey(name);
        model.InitStatic(modelBlob.Data(), modelBlob.Length());
    }


    void FillFactorVector(TVector<float>& result) {
        for (float& f : result) {
           f = RandomNumber<float>();
        }
    }
}

Y_UNIT_TEST_SUITE(FactorsInfluenceTest) {
    void TestSomeDiff(const TString& modelName) {
        TMnSseInfo model;
        NTestSuiteTAnalysisTest::SetModel(model, modelName);
        TVector<float> test(model.GetNumFeats(), 0);
        TSet<size_t> index;
        SetRandomSeed(2);
        NTestSuiteTAnalysisTest::FillFactorVector(test);
        TVector<float> testConstant = test;
        TSet<ui32> usedFactors;
        model.UsedFactors(usedFactors);
        for (int i = 0; i < 10; ++i) {
            size_t nowIndex = RandomNumber<size_t>() % test.size();
            if (usedFactors.find(nowIndex) != usedFactors.end()) {
                index.insert(nowIndex);
                test[nowIndex] = 1.1;
                testConstant[nowIndex] = -0.1;
            } else {
                --i;
            }
        }
        TInfluenceCalcOptions option;
        TVector<float> resultFunction = GetFactorInfluenceRatings (model, test, testConstant, option);
        for (size_t i : xrange(resultFunction.size())) {
            if (index.find(i) == index.end()) {
                UNIT_ASSERT_VALUES_EQUAL(resultFunction[i], 0);
            } else {
                UNIT_ASSERT_UNEQUAL(resultFunction[i], 0);
            }
        }
    }

    void TestOneChange(const TString& modelName) {
        TMnSseInfo model;
        NTestSuiteTAnalysisTest::SetModel(model, modelName);
        TVector<float> testConstant(model.GetNumFeats(), 0);
        SetRandomSeed(3);
        NTestSuiteTAnalysisTest::FillFactorVector(testConstant);
        TSet<ui32> usedFactors;
        model.UsedFactors(usedFactors);
        float eps = 0.0035;
        for (int i = 0; i < 30; ++i) {
            size_t nowIndex = RandomNumber<size_t>() % testConstant.size();
            if (usedFactors.find(nowIndex) == usedFactors.end()) {
                --i;
                continue;
            }
            TVector<float> testFirst = testConstant;
            TVector<float> testSecond = testConstant;
            testFirst[nowIndex] = 1.1;
            testSecond[nowIndex] = -0.1;
            TInfluenceCalcOptions option;
            option.UseAbsoluteInfluence = false;
            TVector<float> resultFunction = GetFactorInfluenceRatings (model, testFirst, testSecond, option);
            UNIT_ASSERT_DOUBLES_EQUAL(resultFunction[nowIndex], (model.DoCalcRelev(testFirst.begin()) - model.DoCalcRelev(testSecond.begin())), eps);

            for(auto j : xrange(resultFunction.size())) {
                if (j == nowIndex) {
                    continue;
                }
                UNIT_ASSERT_VALUES_EQUAL(resultFunction[j], 0.);
            }
        }
    }

    void TestException(const TString& modelName) {
            TMnSseInfo model;
            NTestSuiteTAnalysisTest::SetModel(model, modelName);
            TVector<float> badSize(model.GetNumFeats() - 10, 0);
            TVector<float> goodSize(model.GetNumFeats(), 0);
            SetRandomSeed(4);
            NTestSuiteTAnalysisTest::FillFactorVector(badSize);
            NTestSuiteTAnalysisTest::FillFactorVector(goodSize);
            TVector<float> result;
            TInfluenceCalcOptions option;
            option.UseAbsoluteInfluence = false;
            UNIT_ASSERT_EXCEPTION(result = GetFactorInfluenceRatings (model, badSize, goodSize, option), yexception);
            UNIT_ASSERT_EXCEPTION(result = GetFactorInfluenceRatings (model, goodSize, badSize, option), yexception);
    }

    Y_UNIT_TEST(SomeDiffRu) {
        TestSomeDiff(NTestSuiteTAnalysisTest::Ru);
    }
    Y_UNIT_TEST(SomeDiffFresh) {
        TestSomeDiff(NTestSuiteTAnalysisTest::Fresh);
    }
    Y_UNIT_TEST(OneChangeRu) {
        TestOneChange(NTestSuiteTAnalysisTest::Ru);
    }
    Y_UNIT_TEST(OneChangeFresh) {
        TestOneChange(NTestSuiteTAnalysisTest::Fresh);
    }
    Y_UNIT_TEST(FailRu) {
        TestException(NTestSuiteTAnalysisTest::Ru);
    }
    Y_UNIT_TEST(TestWebL3_2019_07) {
        TMnSseInfo model;
        NTestSuiteTAnalysisTest::SetModel(model, NTestSuiteTAnalysisTest::WebL3_2019_07);

        NFactorSlices::TFactorDomain baselineDomain, testDomain, outputDomain;
        NFactorSlices::TFactorBorders baselineBorders, testBorders, outputBorders;
        NFactorSlices::DeserializeFactorBorders(TString(
            "web_production[0;1894) web_meta[1894;2340) web_meta_rearrange[2340;2361) rapid_clicks[2361;2498) rapid_pers_clicks[2498;2574) web_fresh_detector[2574;2786) begemot_query_factors[2786;2987)"
        ), baselineBorders);
        NFactorSlices::DeserializeFactorBorders(TString(
            "web_production[0;1895) web_meta[1895;2341) web_meta_rearrange[2341;2362) rapid_clicks[2362;2499) rapid_pers_clicks[2499;2575) web_fresh_detector[2575;2787) begemot_query_factors[2787;2988)"
        ), testBorders);
        baselineDomain = NFactorSlices::TFactorDomain(baselineBorders);
        testDomain = NFactorSlices::TFactorDomain(testBorders);

        NMatrixnet::TModelInfo modelInfo = model.Info;
        TString* modelSlices = modelInfo.FindPtr("Slices");
        Y_ENSURE(modelSlices, "not implemented on models without slices");
        NFactorSlices::DeserializeFactorBorders(*modelSlices, outputBorders);
        outputDomain = NFactorSlices::TFactorDomain(outputBorders);

        TFactorStorage baselineStorage(outputDomain);
        TFactorStorage testStorage(outputDomain);
        {
            TVector<float> baselineFactors = {
                0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.447059,0.565217,0,0.166822,0,1,0,0,0,0,0.460375,0,1,0,0,0,0,0,0,0,0,0.4995,0,0,0,0,1,1,1,1,0,0.000314268,0,0,0,0,0,0,0,0,0,0,0,0,0,0.2,0.5,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.000112784,0,0.282353,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.6,0.345098,0,0,0.956863,0.976471,1,0,0,0,0.0745098,0.00784314,1,0.964706,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0.54955,0,0,0,0.0576923,0.0408163,0.000314268,0,0.000649773,0,0.00258398,0,0.00323291,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.533333,0,0,0,0,0,0.996078,1,0,0,0.375,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,1,0.001,0.018309,0.0546179,1,0,0.976471,0.0431373,1,0.290196,0.290196,0.364706,0.905882,1,1,0.615686,0.666667,0.721569,0.933333,0,0.999999,0.999935,0.999637,0.999705,0,0,0,0.156863,0,0.156863,0,0,0,0,0,1,0,0,0,0,0,0.500055,0.500055,0.564706,0.694118,0.972549,0,0,0,0,0,0.375,0,0,0,0,0,0,0,0,0,0,0,0,0,0.203922,0.666667,0.627451,0.984314,0,0,0,1,0,1,0,0,0,0,0,0,0,0,0,1,0.949999,0.05,0,1,0,0,0,1,1,0.54902,0.54902,0.341176,0.776471,0.0156863,0.101961,0.133333,0.993373,0.0831479,0,0,0,0,0,0,0,0,0,0.843137,0,0.798149,0.677082,0.594889,0.602116,0.0579975,0.0409268,0,0.517647,0.580392,0.109804,0.960784,0.898039,0.109804,0.0980392,0.113725,0.917647,1e-12,1e-12,1e-12,1e-12,0,0,0.466694,0.875278,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.0196078,0.0156863,0.00784314,0.180392,0.0156863,0,0,0,0,0,0.0117647,0,0,0,0.0196078,0,0.949999,0.05,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.408284,0.425287,0,0.788235,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.0784314,0.0392157,0.137255,0,0,0,0,0,0,0,0,0,0.486275,0,0,0,0,0,0.583333,0,0,0,0,0,0,0.0666667,0,1,0,0,0,0,0.607843,0.513726,0,0,0,0,0,0.0828495,0,0,0,0,0,0,0.0833333,0.428571,0.0457622,0,0,0.776471,0,0,0,1,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0.275535,0,0,0,0,0,0,0,0,0.619608,0,0,0,0.0156863,0.054902,0,0.486275,0.509804,0.00784314,0,0,0.00392157,0,0,0,0,0.654902,0,0,0,0,0,0,0.678431,0.607843,0,0,0,0,1,0,0.246735,0.0146306,0,0,0,0,0,0,0,0,0,0,0,0.00784314,0.0392157,0,0,0.0196078,0,0.431373,0.87451,0.854902,0.0235294,0.67451,0.831373,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0.713726,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.0458333,0,0,0,0.260522,0,0,0,0,0,0.15,0,0,0,0,0,0,0.156863,0.717647,0.996078,0.960784,0.14902,0,0,0,1,0,0,0,0,0,1,1,1,0.00392157,0,0,0,0,0,0.430055,0,0,0,0,0,0,0,0.301309,0.0783808,0.431483,0,0,0.0431373,0,0,0,0,0,0,0,0,0.5103,0,0,0,1,0,0,0,0,0,0.384314,0,0,0,0,0,0,0,0,0.0156863,0,0,0.546255,0.694118,0,0,0,1,0.109804,0.141176,0,0,0,0,0.184314,0.309804,0.00392157,0,0,0,0.0352941,0.333333,0.0576923,0.00258398,0,0,0,0,0,0.12549,0.372549,0.282353,0,0,0.000314268,0.000314268,1,0,0,0.999499,0.5,0.715742,1,1,1,0,0,0.988235,0,0.999093,0,0,0.176471,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.0823529,0,0,0,0,0.760784,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.368627,0,0,0.576471,0,0,0,0,0,0,0,0,0,0,0,0,0,0.705882,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.705882,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0.156863,0.156863,0.219186,0.926894,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.996518,0.00790743,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.305882,0.999489,0.999427,0.99987,1,0.909091,0.929412,0.686275,0.383253,0.0517181,0.0398474,0.135231,0.929412,0.892934,0.980191,0.0526611,0.35246,0.32895,0.32895,0.345509,0.68603,0.368655,0.998871,0,0,0,0,0,0,0,0,0,0,0.342904,0,0,0,1,0,0,0.399471,0.375,0,0,0,0,0,0,0,0,0,0,0.772549,0.713726,0.286275,0.360784,0.513726,0.490196,0.529412,0.0352941,0.239216,0.235294,0.113725,0.968628,0.00392157,0.498039,0.207843,0,0,0,0,0,0,0.99982,0.996365,0.998335,0,0,0,0,0,0.698039,0.698039,0.698039,0.698039,0.816275,0.731373,0.498418,0.731373,0.698039,0.698039,0.731373,0.999984,0.91089,0.59228,0.392952,0.603922,0.603922,0.603922,0.998347,0.995874,0.999903,0.09231,0.0714199,0.101961,0.101961,0.101961,0.101961,0.101961,0.101961,0.101961,0.98992,0.990288,0.97605,0.998925,0.509895,0.929412,0.929412,0.929412,0.929412,0.929412,0.929412,0.929412,0.887773,1,0.570163,0.999007,0.929412,0,0.219998,0,0,0.582353,0.282219,0,0,0,0,0,0,0.481626,0.196918,0.705882,0.13776,0.0970588,0.705882,0.705882,0.987285,0.996788,0.792157,0.611765,0.611765,0.611765,0.611765,0.905944,0.629583,0.929412,0.705882,0.705882,0,0,0,0.482696,0.686275,0.686275,0.686275,0.643777,0.512173,0.99981,0.993806,0.74902,0.74902,0.74902,0.74902,0.689916,0.656863,0,0,0,0,0,0,0,0,0,0,0,0.74902,0.488771,0.285583,0.309117,0.74902,0.801771,0.134256,0.698039,0.808759,0.421908,0.282475,0.203549,0.255853,0.40908,0.564512,0.345191,0,1,0,0,0,0,1,0,0,0.698039,0.698039,0.698039,0.698039,0.664706,0.664706,0.647441,0.401429,0,0,0,0,0,0,0.999994,0.745098,0.745098,0.745098,0.99934,0,0,0.611765,0.686275,0.686275,0.929412,0.929412,0.101961,0.99982,0.996365,0.998335,0,0,0,0,0,0.375,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0.74902,0.324117,1,1,1,1,0,0,0,0.000772231,0.000510075,0.790475,0.713725,0,0.758862,0,0.929412,0.929412,0.929412,0.185882,0.509895,0,0,0,0.701961,0.541176,0,0.835294,0.792157,0.670588,0,0.00392157,0.0313726,0.67451,0.564706,0.788235,0.980392,0.541176,0.809122,0,0.730044,0,0.894118,0.528628,0.884467,0,0.462569,0.443137,0.479216,0.443137,0.443137,0.443137,0.369944,0.664706,0.687231,0.313726,0.5,0.5,0.5,0.5,0.329051,0.135953,0,0,0.268052,0.0412935,0,0.0509804,0.0627451,0.541176,0.752941,0.152941,0.858824,0.490196,0.2,0.356863,0.615686,0.12549,0.105882,0.796094,0.633082,0,0,0,0.847059,0.352941,0.588235,0.397669,0.386632,0,0.999291,0,0.412325,0.51158,0,0.998703,0.833179,0.324235,0,0.447693,0.0017989,1,0.779911,0.954174,0.559607,0.734465,0.738328,0.484618,0.484618,0,0.484618,0.484618,0.500733,0.500733,0,0.500733,0.500733,1,0.894118,0.368627,0.999987,1,0.996078,0.999248,1,0,0,0,0,0,0,0,0.0117647,0.792157,0.682353,0.360784,0.0235294,0,0,0.2,0.196078,0.0196078,0.356863,0.0666667,0.411765,0.0196078,0,0,0.439216,0.921569,0.101961,0.054902,0.572549,0.129412,0,0.0470588,0.105882,0.282353,0,0,0,0,0,1,0,0.0371094,0.115234,0.447266,0,0.684941,0,0,0.689852,1,1,1,1,0.908996,0.893115,0.885409,0.144782,0.971013,0.15741,0.83466,0.81267,0.830078,0.965996,0.96411,0.798079,0.934902,0.845429,0.0127819,0.876298,0.863373,0.756963,0.649037,0.0280904,0,0.996078,0.996078,0.951373,0.996078,0.996078,0.320221,0.70963,0,0,0.304853,0.474584,0.463961,0.539614,0.524868,0.458878,0,0,0,0.0140856,0,0,0.478898,0.484365,0,0,0.74161,0.00392157,0.0196078,0.32549,0.0196078,0.133333,0.0745098,0.0666667,0.0509804,0.505882,0.239216,0,0.282353,0.47451,0.0392157,0.266667,0,0,0,0.403043,0,0.768627,0.643137,0.337255,0.0627451,0.00784314,0.0352941,0.231373,0.113725,0.0117647,0.313726,0.0392157,0.513726,0.0156863,0,0,0,0.909804,0.0980392,0.0745098,0.470588,0.0588235,0,0.0156863,0.0862745,0.592157,0.0705882,0.113725,0.160784,0.0627451,0.998335,0.996365,1,0.74902,0,0.998335,0.998335,0.99982,0.99982,0.929412,0.99982,0.99982,0,0.99982,0.705882,0,0.200073,0,0,0,0,0,0,0.592095,0.977031,0.036257,0,0,0,0,0,0,0,0.492064,0.492064,0,0,0,0,0,0,0,0,0,0,0,1,1,0.0909091,1,0,0.906545,0.892134,0,0,0.262745,0,0,0.72549,0.52549,0.705882,0.113725,0.4,0.0509804,0.180392,0,0,0.435294,0,0.0392157,0,0.192157,0.87451,0.196078,0,0.0392157,0.984314,0.862745,0,0.47451,0,0.137255,0.666667,0,0,0,0.929412,0.227451,0.0313726,0,0.321569,0.596078,0.0352941,0.588235,0.568627,0.423529,0.627451,0.0588235,0.521569,0,0,0.741842,0,0,0,0,1,1,0,1,0.770078,0.507503,0.341965,0.0170773,0.0356965,0.484645,0,0.0217965,0.355411,0.93,0,0,0.411765,1,0,0.792157,0.360784,0.521569,0.372549,0.701961,0.894118,1,0,0,0,0,0,0,0,0,0,0,0,0,0.180392,0.180392,0.223529,0.168627,0,0.223529,0.2,0.266667,0.133333,0.121569,0.984314,0.564706,1,0.054902,0.996078,0.945098,0.372549,0.286275,0.894118,0.745098,0.282353,0.560784,1,0.905882,0.501961,0.843137,0.486275,0.964706,0.917647,0.796079,1,0.305882,0.941177,0.572549,0.631373,0.94902,0.301961,0.0784314,0.00392157,0,0.00784314,0.113725,0.52549,0.94902,0.0196078,0.815686,0.905882,1,0,0.428571,0.0186157,0,0.123923,0,0,0,0.702308,0.0627451,0.533333,0.11811,0,0.142857,0,0,0,0,0.415686,0,0.992157,0.984314,0.941177,0.74902,0.427451,0.156863,0.0941177,0,0,0.317647,0.435294,0.588235,0.83397,0.375,0.0483092,0.145098,0.807843,0.831373,0.498039,0.482353,0.0470588,0.639216,0.592157,0.282353,0.282353,0.247059,0.980392,0.0901961,0.541176,0.105882,0.054902,0.529412,0.180392,0.941177,0.180392,0.0666667,0.933333,0,0.12549,0.611765,0.168627,0.113725,0,0.654902,0.129412,0,1,0.34719,0.812535,0.524431,0.329412,0.988223,0.999488,0.427481,0.276543,0,0.983182,0,0,0.227053,0.549224,0,0.996509,0.998335,0,0,0.99629,0,0,0,0.621036,0.048927,0,0.0585876,0.547735,0.146454,0,0,0,0.99982,0.686275,0.309263,0.28132,1,0,0.372549,0.811765,-0.00891142,0.0512273,0,0,0,0,0,0.515304,0.516081,0,0.530292,0.056287,0.398549,0.683934,0,0,0,0,0,0,0.694098,0,0.976526,0.0234742,0.757489,0,0,0,0,0,0.111111,0.609352,0.58216,0,0,0.631373,0.776471,0.705882,0,0.480894,1,0,1,0,0.988193,0.78557,0.032292,0.524556,0,0.0685189,0.681755,0.0296425,0.266141,0.244262,0,0.000229964,0.0128205,0,1,1,0.2,0.0714286,0,0,0.2,0.037037,0.047619,0.166667,1,1,0,1,0,0,0,0,0.0243902,0.5,0.00714286,0.333333,0,0.2,0,1,0,0,0,0,0,0.999614,0,0.954423,0,0,0.0909091,0.0909091,0,0.00943396,0.792157,0.788235,0,0,0,0,0,0,0.0870096,0.0689655,0.852941,0.706445,0.117789,0.285714,0.130435,0.406858,0.375,0.431483,0.00784314,0,0,0,0,0.351987,0.0825034,0,0.177163,0.588235,0,0.312374,0,0,0,0.00938967,0.788618,0,0.12626,0.905621,0.0616552,0.220337,0.0408064,0.144444,0.999978,0.995273,0.028169,0.74902,0.028169,0.74902,0.144312,0.0985916,0.132986,0.028169,0.122748,0,0,0,0,0,0.762241,0.103673,1,0.22002,0.9613,0,0.976526,0,0.981221,0.159624,0.971831,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.62676,0.62676,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0.631455,0.137295,0.130219,0.746109,0.136946,0.435413,0.0168394,0.879443,0.0822115,0.106001,0.422535,0.00327118,0.153825,0.875546,1,0.225224,0,0,0,0.958337,0.18426,0.1017,0,0,0.999893,0,0.503476,0,0.929412,1,0,0,0,0,0,0,0,0,0,0,0,0.484351,0,0.252275,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.00469484,0.00469484,1,0.00938967,0,0,0,0,0.589525,0,0,0,0.5,0.572596,0,0.230769,0.175965,0.0117801,0.261777,0.235025,0,0.1258,0.0757519,0.0689655,0.804224,0.130435,0.428571,0.419217,0.555556,0.999889,0.966467,0,0,0.93525,0,0,0.571429,0,0,0.999878,0,0.0449212,0.553661,0.972048,0.342192,0.888234,0.957762,0,0.5,0.278431,0.963154,0.869971,0.944268,0.99981,0.977216,0.158443,0.667208,0.0416701,0.0525543,0.0490188,0.05784,0.065144,0.0521896,0.0344008,0.0380965,0.0493972,0.0474776,0.0591163,0.0230088,0.0821251,0.0591163,0.034573,0.0336497,0.0360931,0.715463,0,0,0,0,0,0.0267046,0.0702324,0.0340427,0.042057,0,0.0413625,0,0.634883,0.0789666,0.0263463,0.000119448,0.26532,0.00469484,0.977383,0.0227792,0.119249,0.608656,0.5,0.416082,0.409358,0.705882,0,0.886098,0.0319226,0.581411,0.277187,0.889746,0.402052,0.55893,0,0.0234742,0.0403222,0.037823,0,0.0301615,0.334672,0,0.839696,0.189783,0.795031,0.999889,0.461891,0.0564778,0.999675,0.572783,0.995779,0.236789,0,0,0.0526868,0.99491,0.426148,0,0.693563,1,0.717739,0.0948187,0.272351,0.26662,0,0.344127,0.999999,0.498039,0.741135,0.0900474,0.579991,0.712102,0.0352941,0.99491,0.0840855,0,0.922371,0.73399,0.70153,0.693027,0.97999,0.11506,0.678431,0,0.489367,0.00943396,0.142857,0,0.797532,0.831693,0.455271,0.0811199,0.0224347,0.0697674,0.106514,0.0529042,0.101754,0.771571,0.406521,0.0187793,0.506675,0.0390643,0,0.322704,0.193333,0.995874,0.99934,0.624198,0.288889,0.119353,0.668443,0.4,0.417255,0.0776471,0.0831373,0.883137,0.607843,0.126755,0.138718,0.261035,0.931149,0.982107,0.000304494,0.16391,0,0.594487,0,0.377205,0.983369,0.524194,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.889912,0.000584845,0.314506,0.48145,0.082889,0.0881661,0.362517,0.554947,0.36365,0.518625,0.0794807,0.538664,0.576329,0.089517,0.0757004,0.573832,0.520277,0.486896,0.889912,0.462875,0.082889,0.103243,0.511774,0.536803,0.5,0.0885516,0.538699,0.0972818,0.0823379,0.574117,0.0630767,0.570462,0.74919,0,0,0.5,0,0,0,0.5,1,0,0.5,0,0.5,0,1,0,0.5,0,0.5,0,0.5,0,0,0,0.5,0,0.000284467,0.331598,0.777157,0.0969949,0.331598,0.0628347,0.804113,0.0665708,0.631697,0.683271,0.0761588,0.520277,0.000388132,0.554672,0.511774,0.000284467,0.590907,0.613808,0.626462,0.74919,0.972252,0.97535,0.978449,0.976484,0.976762,0.97704,0.975718,0.979196,0.982673,0.9792,0.979511,0.979823,0.978224,0.97849,0.978585,0.978628,0.97867,0.979181,0.979386,0.979591,0.382666,0.39084,0.399014,0.517372,0.518241,0.51911,0.513979,0.514055,0.514131,0.510871,0.51295,0.514478,0.514569,0.514661,0.514835,0.514905,0.514975,0.51432,0.514286,0.473743,0.474018,0.474292,0.857584,0.802854,0.623329,1,0.149215,0,0.706724,1,1,0.089235,0.0891848,0.854233,0.95473,0.0874669,0.917675,0.0883949,0.920804,0,0.5,1,0,0,0,0.5,0,0,0.5,1,0,0.5,1,0,0,0,0,0,0,0,0.5,0,0,0.5,0,0,0.5,0,0,0.5,0,0,0.5,1,0,0,0,0,0.5,0.5,0.5,0,0,0,0,0,0,0.5,0.5,0.5,0,0,0,0,0,0,0.5,0.5,0.5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.163265,0.00471698,0.00903523,0.000304969,0.0190507,0.000401389,0.153846,0.705882,0.999978,0.976415,0.144651,0.428136,0.0323681,0,0,0.0458333,0,0,0,0,1,0,0,0,1,0,0,0,0,0.583333,0,0,1,0.0828495,0.431483,1,0,0,0,0,1,0,0.949999,0.05,0,0.949999,0.05,0,0,0.0783808,0.301309,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.15,0.397669,0.386632,0.5103,0.00790743,0.342904,0,0,0,0,1,0,0,0,0,0.399471,0.329051,0.135953,0.268052,0.0412935,0,0.0117647,0.792157,0.682353,0.360784,0.0235294,0,0,0.2,0.196078,0.0196078,0.356863,0.0666667,0.411765,0.0196078,0,0.0862745,0.439216,0.921569,0.101961,0.054902,0.572549,0.129412,0,0.0470588,0.105882,0.282353,0,0,0,0,0.304853,0.474584,0.463961,0.539614,0.524868,0.458878,0.403043,0,0.768627,0.643137,0.337255,0.0627451,0.00784314,0.0352941,0.231373,0.113725,0.0117647,0.313726,0.0392157,0.513726,0.0156863,0,0.109804,0.427451,0.909804,0.0980392,0.0745098,0.470588,0.0588235,0,0.0156863,0.0862745,0.592157,0.0705882,0.113725,0.160784,0.0627451,0.200073,0.260522,0,0.906545,0.892134,0.262745,0,0,0.72549,0.52549,0.705882,0.113725,0.4,0.0509804,0.180392,0.741842,0.770078,0.507503,0.341965,0.0170773,0.0356965,0.484645,0,0.0217965,0.355411,1,0,0,0,0,0.0458333,0,0,0,0,1,0,0,0,1,0,0,0,0,0.583333,0,0,1,0.0828495,0.431483,1,0,0,0,0,1,0,0.949999,0.05,0,0.949999,0.05,0,0,0.0783808,0.301309,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.15,0.397669,0.386632,0.5103,0.00790743,0.342904,0,0,0,0,1,0,0,0,0,0.399471,0.329051,0.135953,0.268052,0.0412935,0,0.0117647,0.792157,0.682353,0.360784,0.0235294,0,0,0.2,0.196078,0.0196078,0.356863,0.0666667,0.411765,0.0196078,0,0.0862745,0.439216,0.921569,0.101961,0.054902,0.572549,0.129412,0,0.0470588,0.105882,0.282353,0,0,0,0,0.304853,0.474584,0.463961,0.539614,0.524868,0.458878,0.403043,0,0.768627,0.643137,0.337255,0.0627451,0.00784314,0.0352941,0.231373,0.113725,0.0117647,0.313726,0.0392157,0.513726,0.0156863,0.109804,0.909804,0.0980392,0.0745098,0.470588,0.0588235,0,0.0156863,0.0862745,0.592157,0.0705882,0.113725,0.160784,0.0627451,0.200073,0.260522,0,0.906545,0.892134,0.262745,0,0,0.72549,0.52549,0.705882,0.113725,0.4,0.0509804,0.180392,0.741842,0.770078,0.507503,0.341965,0.0170773,0.0356965,0.484645,0,0.0217965,0.355411,0,0,0,0.83397,0.375,0.0483092,0,0,0,0,0,0,0,0,0,0.227053,0,0.515304,0.516081,0,0.530292,0.056287,0.398549,0,0};
            TVector<float> testFactors = {
                0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.447059,0.555556,0,0.166822,0,1,0,0,0,0,0.460779,0,1,0,0,0,0,0,0,0,0,0.4995,0,0,0,0,1,1,1,1,0,0.000273598,0,0,0,0,0,0,0,0,0,0,0,0,0,0.2,0.5,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.000112784,0,0.282353,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.560784,0.337255,0,0,0.956863,0.976471,1,0,0,0,0.0745098,0.00784314,1,0.964706,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0.552573,0,0,0,0.0483871,0.0408163,0.000273598,0,0.000487329,0,0.00384615,0,0.00433255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.533333,0,0,0,0,0,0.996078,1,0,0,0.375,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,1,0.001,0.018309,0.0546179,1,0,0.976471,0.0431373,1,0.290196,0.290196,0.364706,0.905882,1,1,0.619608,0.666667,0.721569,0.933333,0,0.999999,0.999937,0.999648,0.999741,0,0,0,0.156863,0,0.156863,0,0,0,0,0,1,0,0,0,0,0,0.500056,0.500056,0.564706,0.694118,0.972549,0,0,0,0,0,0.375,0,0,0,0,0,0,0,0,0,0,0,0,0,0.203922,0.666667,0.627451,0.984314,0,0,0,1,0,1,0,0,0,0,0,0,0,0,0,1,0.949999,0.05,0,1,0,0,0,1,1,0.54902,0.54902,0.341176,0.776471,0.0156863,0.101961,0.133333,0.993373,0.0860663,0,0,0,0,0,0,0,0,0,0.87451,0,0.804766,0.677082,0.594348,0.632837,0.0486541,0.0409268,0,0.52549,0.580392,0.109804,0.956863,0.901961,0.0745098,0.0980392,0.113725,0.898039,1e-12,1e-12,1e-12,1e-12,0,0,0.439774,0.869829,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.0196078,0.00784314,0.00784314,0.188235,0.0235294,0,0,0,0,0,0.0117647,0,0,0,0.0196078,0,0.949999,0.05,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.408284,0.459459,0,0.788235,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.0784314,0.0392157,0.137255,0,0,0,0,0,0,0,0,0,0.486275,0,0,0,0,0,0.583333,0,0,0,0,0,0,0.0666667,0,1,0,0,0,0,0.607843,0.513726,0,0,0,0,0,0.0828495,0,0,0,0,0,0,0.0833333,0.428571,0.0457622,0,0,0.776471,0,0,0,1,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0.278683,0,0,0,0,0,0,0,0,0.619608,0,0,0,0.0156863,0.054902,0,0.486275,0.490196,0.00784314,0,0,0.00392157,0,0,0,0,0.654902,0,0,0,0,0,0,0.686275,0.607843,0,0,0,0,1,0,0.246735,0.0146306,0,0,0,0,0,0,0,0,0,0,0,0.00784314,0.0392157,0,0,0.0901961,0,0.431373,0.87451,0.854902,0.0235294,0.678431,0.831373,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0.713726,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.0458333,0,0,0,0.260522,0,0,0,0,0,0.15,0,0,0,0,0,0,0.133333,0,0.996078,0.964706,0.14902,0,0,0,1,0,0,0,0,0,1,1,1,0.00392157,0,0,0,0,0,0.429739,0,0,0,0,0,0,0,0.301309,0.0783808,0.431483,0,0,0.0431373,0,0,0,0,0,0,0,0,0.5103,0,0,0,1,0,0,0,0,0,0.384314,0,0,0,0,0,0,0,0,0.0156863,0,0,0.546255,0.572549,0,0,0,1,0.109804,0.141176,0,0,0,0,0.184314,0.196078,0.00392157,0,0,0,0.0352941,0.333333,0.0483871,0.00384615,0,0,0,0,0,0.12549,0.372549,0.282353,0,0,0.000273598,0.000273598,1,0,0,0.999289,0.5,0.715742,1,1,1,0,0,0.988235,0,0.999093,0,0,0.176471,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.0352941,0,0,0,0,0.752941,0.996078,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.364706,0,0,0.576471,0,0,0,0,0,0,0,0,0,0,0,0,0,0.603922,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.603922,0.996078,0,0,0,0,0,0,0,0,0,0,0,0,0,0.133333,0.133333,0.203343,0.876063,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.995934,0.00790743,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.305882,0.999513,0.999445,0.99987,1,0.909091,0.929412,0.647059,0.400104,0.0517181,0.0398474,0.135231,0.929412,0.893645,0.980098,0.0526611,0.35249,0.32895,0.32895,0.345509,0.68603,0.368655,0.998906,0,0,0,0,0,0,0,0,0,0,0.342904,0,0,0,1,0,0,0.399471,0.375,0,0,0,0,0,0,0,0,0,0,0.772549,0.713726,0.286275,0.360784,0.513726,0.490196,0.529412,0.0352941,0.239216,0.235294,0.113725,0.968628,0.00392157,0.498039,0.207843,0,0,0,0,0,0,0.99982,0.996396,0.99832,0,0,0,0,0,0.537255,0.537255,0.537255,0.537255,0.793305,0.65098,0.475983,0.65098,0.537255,0.537255,0.65098,0.999984,0.911478,0.598789,0.396794,0.607843,0.607843,0.607843,0.998358,0.995901,0.999903,0.09231,0.0714199,0.101961,0.101961,0.101961,0.101961,0.101961,0.101961,0.101961,0.98992,0.990288,0.97605,0.998925,0.509895,0.929412,0.929412,0.929412,0.929412,0.929412,0.929412,0.929412,0.888328,1,0.570432,0.999006,0.929412,0,0.219998,0,0,0.582353,0.282219,0,0,0,0,0,0,0.481626,0.12709,0.603922,0.0869654,0.0571895,0.603922,0.603922,0.985638,0.996368,0.792157,0.596078,0.596078,0.596078,0.596078,0.908163,0.610195,0.929412,0.690196,0.690196,0,0,0,0.482696,0.647059,0.647059,0.647059,0.620602,0.512125,0.99981,0.993806,0.74902,0.74902,0.74902,0.74902,0.691472,0.656863,0,0,0,0,0,0,0,0,0,0,0,0.74902,0.526828,0.31514,0.318754,0.74902,0.883983,0.137024,0.537255,0.808759,0.421908,0.282475,0.203549,0.255853,0.40908,0.564512,0.345191,0,1,0,0,0,0,1,0,0,0.698039,0.698039,0.698039,0.698039,0.664706,0.664706,0.657168,0.406143,0,0,0,0,0,0,0.999993,0.733333,0.733333,0.733333,0.999335,0,0,0.596078,0.647059,0.647059,0.929412,0.929412,0.101961,0.99982,0.996396,0.99832,0,0,0,0,0,0.375,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0.74902,0.324117,1,1,1,1,0,0,0,0.000772231,0.000510075,0.790475,0.713725,0,0.758862,0,0.929412,0.929412,0.929412,0.185882,0.509895,0,0,0,0.698039,0.541176,0,0.835294,0.792157,0.670588,0,0.00392157,0.0313726,0.67451,0.564706,0.792157,0.980392,0.541176,0.809122,0,0.781275,0,0.956863,0.568627,0.950107,0,0.452505,0.435294,0.46902,0.435294,0.435294,0.435294,0.369944,0.664706,0.687231,0.313726,0.5,0.5,0.5,0.5,0.329051,0.135953,0,0,0.268052,0.0412935,0,0.0509804,0.0627451,0.541176,0.752941,0.152941,0.858824,0.490196,0.2,0.356863,0.615686,0.12549,0.105882,0.796094,0.633082,0,0,0,0.847059,0.352941,0.588235,0.397669,0.386632,0,0.999316,0,0.412325,0.511484,0,0.998703,0.833187,0.324235,0,0.447693,0.00177931,1,0.779913,0.978853,0.559607,0.734465,0.738328,0.484618,0.484618,0,0.484618,0.484618,0.500733,0.500733,0,0.500733,0.500733,1,0.894118,0.368627,0.999986,1,0.87451,0.999248,1,0,0,0,0,0,0,0,0.0117647,0.792157,0.682353,0.360784,0.0235294,0,0,0.2,0.196078,0.0196078,0.356863,0.0666667,0.411765,0.0196078,0,0,0.439216,0.921569,0.101961,0.054902,0.572549,0.129412,0,0.0470588,0.105882,0.282353,0,0,0,0,0,1,0,0.0371094,0.115234,0.447266,0,0.684941,0,0,0.689852,1,1,1,1,0.908996,0.894037,0.885553,0.144932,0.971562,0.157499,0.83466,0.81267,0.830078,0.965996,0.96411,0.798079,0.934902,0.845429,0.0127819,0.876298,0.863373,0.756963,0.649037,0.0280904,0,0.752941,0.87451,0.902745,0.752941,0.752941,0.271594,0.70963,0,0,0.304853,0.474584,0.463961,0.539614,0.524868,0.458878,0,0,0,0.0141438,0,0,0.478922,0.484365,0,0,0.74161,0.00392157,0.0156863,0.321569,0.0196078,0.133333,0.0784314,0.0705882,0.0509804,0.505882,0.231373,0,0.301961,0.478431,0.0392157,0.270588,0,0,0,0.403043,0,0.768627,0.643137,0.337255,0.0627451,0.00784314,0.0352941,0.231373,0.113725,0.0117647,0.313726,0.0392157,0.513726,0.0156863,0,0,0,0.909804,0.0980392,0.0745098,0.470588,0.0588235,0,0.0156863,0.0862745,0.592157,0.0705882,0.113725,0.160784,0.0627451,0.99832,0.996396,1,0.74902,0,0.99832,0.99832,0.99982,0.99982,0.929412,0.99982,0.99982,0,0.99982,0.690196,0,0.200073,0,0,0,0,0,0,0.592095,0.977031,0.036257,0,0,0,0,0,0,0,0.492064,0.492064,0,0,0,0,0,0,0,0,0,0,0,1,1,0.0909091,1,0,0.906545,0.892134,0,0,0.262745,0,0,0.72549,0.52549,0.705882,0.113725,0.4,0.0509804,0.180392,0,0,0.439216,0,0.0392157,0,0.196078,0.87451,0.196078,0,0.0392157,0.984314,0.862745,0,0.47451,0,0.137255,0.670588,0,0,0,0.929412,0.227451,0.0352941,0,0.321569,0.588235,0.0352941,0.588235,0.568627,0.423529,0.627451,0.0588235,0.517647,0,0,0.741842,0,0,0,0,1,1,0,1,0.770078,0.507503,0.341965,0.0170773,0.0356965,0.484645,0,0.0217965,0.355411,0.93,0,0,0.411765,1,0,0.792157,0.360784,0.521569,0.372549,0.701961,0.894118,1,0,0,0,0,0,0,0,0,0,0,0,0,0.172549,0.180392,0.223529,0.172549,0,0.223529,0.203922,0.258824,0.133333,0.121569,0.984314,0.576471,1,0.054902,0.996078,0.945098,0.372549,0.286275,0.898039,0.745098,0.282353,0.560784,1,0.886275,0.501961,0.843137,0.486275,0.964706,0.917647,0.792157,1,0.305882,0.941177,0.572549,0.631373,0.94902,0.301961,0.0784314,0.00392157,0,0.00784314,0.109804,0.52549,0.94902,0.0196078,0.823529,0.909804,1,0,0.428571,0.0186157,0,0.123923,0,0,0,0.701974,0.0588235,0.533333,0.11811,0,0.142857,0,0,0,0,0.419608,0,0.996078,0.984314,0.945098,0.752941,0.462745,0.156863,0.0941177,0,0,0.317647,0.435294,0.588235,0.83397,0.375,0.0483092,0.145098,0.811765,0.831373,0.498039,0.482353,0.0509804,0.643137,0.592157,0.278431,0.270588,0.243137,0.980392,0.0941177,0.545098,0.105882,0.054902,0.52549,0.180392,0.941177,0.180392,0.0627451,0.933333,0,0.12549,0.619608,0.172549,0.109804,0,0.654902,0.129412,0,1,0.34719,0.812535,0.524431,0.329412,0.988223,0.999488,0.427481,0.276543,0,0.983182,0,0,0.227053,0.549224,0,0.99657,0.99832,0,0,0.996344,0,0,0,0.621036,0.048927,0,0.0585876,0.547735,0.144528,0,0,0,0.99982,0.647059,0.309263,0.281317,1,0,0.372549,0.811765,-0.00671422,0.0408263,0,0,0,0,0,0.515304,0.516081,0,0.530292,0.056287,0.398549,0.683934,0,0,0,0,0,0,0.694098,0,0,0.976636,0.0233645,0.763973,0,0,0,0,0,0.107532,0.611856,0.584112,0,0,0.631373,0.776471,0.603922,0,0.481354,1,0,1,0,0.988249,0.790159,0.0323332,0.517111,0,0.0966736,0.697125,0.0278112,0.254452,0.248554,0,0.000113688,0.0126582,0,1,1,0.2,0.0714286,0,0,0.2,0.0555556,0.0454545,0.166667,1,1,0,1,0,0,0,0,0.0294118,0.5,0.00724638,0.333333,0,0.25,0,1,0,0,0,0,0,0.999395,0,0.953997,0,0,0,0,0,0.00471698,0.792157,0.792157,0,0,0,0,0,0,0.0367197,0.0344828,0.852941,0.709961,0.0538493,0.166667,0.0909091,0.399597,0.393939,0.431483,0.00392157,0,0,0,0,0.351639,0.0718495,0,0.177163,0.588235,0,0.358187,0,0,0,0.00934579,0.771084,0,0.118022,0.894678,0.0684612,0.228147,0.0476304,0.155818,0.999979,0.995144,0.0280374,0.816239,0.0280374,0.816239,0.150636,0.102804,0.141918,0.0135135,0.115432,0,0,0,0,0,0.766829,0.104801,1,0.218667,0.960376,0,0.976636,0,0.981308,0.158879,0.971963,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.53937,0.53937,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0.611371,0.138465,0.122877,0.746109,0.13642,0.439503,0.0158761,0.879443,0.0780145,0.105988,0.420561,0.00318709,0.158713,0.865915,1,0.223114,0,0,0,0.95662,0.184712,0.13113,0,0,0.999893,0,0.404205,0,0.929412,1,0,0,0,0,0,0,0,0,0,0,0,0.484222,0,0.252275,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.0046729,0.0046729,1,0.0046729,0,0,0,0,0,0,0,0,0.499999,0.404207,0,0.230769,0.16301,0.0196078,0.173345,0.0980391,0,0.147942,0.0379559,0.0344828,0.804201,0.0909091,0.428571,0.391735,0.545455,0.999889,0.96653,0,0,0.93525,0,0,0.857143,0,0,0.999883,0,0.0477076,0.593355,0.955657,0.346327,0.888234,0.957756,0,0.5,0.27451,0.963154,0.869971,0.940491,0.99981,0.977216,0.158443,0.667208,0.0416701,0.0525543,0.0490188,0.05784,0.065144,0.0521896,0.0344008,0.0380965,0.0493972,0.0474776,0.0591163,0.0230088,0.0821251,0.0591163,0.034573,0.0336497,0.0360931,0.715463,0,0,0,0,0,0.0267046,0.0702324,0.0340427,0.042057,0,0.0413625,0,0.635102,0.0790495,0.0323438,0.000119388,0.26532,0.0046729,0.977383,0.0227792,0.120451,0.678992,0.5,0.40957,0.40957,0.690196,0,0.886296,0.0312983,0.581213,0.251933,0.870134,0.401371,0.594325,0,0.0140187,0.0403222,0.037823,0,0.0301615,0.339477,0,0.804391,0.18411,0.827104,0.999889,0.493062,0.0528679,0.999672,0.572783,0.995779,0.215568,0,0,0.0504716,0.995779,0.408231,0,0.720519,1,0.719368,0.096594,0.266034,0.228609,0,0.343053,0.999999,0.545098,0.742261,0.0893068,0.502108,0.713388,0.0352941,0.994516,0.0989976,0,0.753763,0.735293,0.70563,0.695821,0.971027,0.110453,0.678431,0,0.486845,0.00469484,0.142857,0,0.797532,0.831693,0.455271,0.0811199,0.0234796,0.0570902,0.0840741,0.0529042,0.0830946,0.771571,0.406559,0.0140187,0.507437,0.0443368,0,0.282734,0.221242,0.995901,0.999335,0.675926,0.361111,0.141501,0.668443,0.5,0.521569,0.0627451,0.0843137,0.902941,0.635294,0.157186,0.167935,0.326294,0.929631,0.979102,0.000520899,0.0776579,0,0.673953,0,0.337461,0.983369,0.524295,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.975175,0.000195634,0.197117,0.341873,0.0402254,0.0296188,0.206864,0.358777,0.322632,0.444758,0.0990054,0.367695,0.393545,0.0597628,0.106023,0.355479,0.360504,0.502397,0.975175,0.358024,0.0453594,0.104763,0.502397,0.375271,0.444522,0.108833,0.367695,0.0597628,0.106023,0.355479,0.110917,0.389929,0.67628,0,0,0.380612,0,0,0,0.380612,0.761223,0,0.380612,0,0.14696,0,0.640326,0,0.492098,0.000785882,0.380612,0,0.380612,0.00115783,0,0.176569,0.5,0,0.0000408844,0.114607,0.501035,0.0398904,0.114607,0.174416,0.601534,0.162916,0.318116,0.357633,0.165781,0.36277,0,0.321435,0.49467,0.0000408844,0.447236,0.317768,0.331368,0.67887,0.970534,0.973841,0.977148,0.977468,0.977743,0.978018,0.971149,0.974765,0.978382,0.978614,0.978918,0.979223,0.978884,0.979147,0.97884,0.978882,0.978924,0.978933,0.979134,0.979335,0.338827,0.348936,0.359045,0.529024,0.529932,0.530841,0.515922,0.515998,0.516074,0.523445,0.525598,0.516516,0.516607,0.516698,0.516751,0.516821,0.516891,0.516212,0.516131,0.47462,0.474887,0.475155,0.945042,0.515452,0.908221,1,0.218942,1,0.312432,1,1,0.082451,0.083529,0.864762,0.957461,0.0857591,0.921643,0.0830656,0.924684,0,0.5,1,0,0,0,0.5,0,0,0.5,1,0,0.5,1,0,0,0,0,0,0,0,0.5,0,0,0.5,0,0,0.5,0,0,0.5,0,0,0.5,1,0,0,0,0,0.5,0.5,0.5,0,0,0,0,0,0,0.5,0.5,0.5,0,0,0,0,0,0,0.5,0.5,0.5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.183673,0.0141509,0.0105692,0.0378148,0.0101359,0.0734371,0.16129,0.603922,0.999979,0.976415,0.151345,0.415845,0.0324854,0,0,0.0458333,0,0,0,0,1,0,0,0,1,0,0,0,0,0.583333,0,0,1,0.0828495,0.431483,1,0,0,0,0,1,0,0.949999,0.05,0,0.949999,0.05,0,0,0.0783808,0.301309,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.15,0.397669,0.386632,0.5103,0.00790743,0.342904,0,0,0,0,1,0,0,0,0,0.399471,0.329051,0.135953,0.268052,0.0412935,0,0.0117647,0.792157,0.682353,0.360784,0.0235294,0,0,0.2,0.196078,0.0196078,0.356863,0.0666667,0.411765,0.0196078,0,0.0862745,0.439216,0.921569,0.101961,0.054902,0.572549,0.129412,0,0.0470588,0.105882,0.282353,0,0,0,0,0.304853,0.474584,0.463961,0.539614,0.524868,0.458878,0.403043,0,0.768627,0.643137,0.337255,0.0627451,0.00784314,0.0352941,0.231373,0.113725,0.0117647,0.313726,0.0392157,0.513726,0.0156863,0,0.109804,0.427451,0.909804,0.0980392,0.0745098,0.470588,0.0588235,0,0.0156863,0.0862745,0.592157,0.0705882,0.113725,0.160784,0.0627451,0.200073,0.260522,0,0.906545,0.892134,0.262745,0,0,0.72549,0.52549,0.705882,0.113725,0.4,0.0509804,0.180392,0.741842,0.770078,0.507503,0.341965,0.0170773,0.0356965,0.484645,0,0.0217965,0.355411,1,0,0,0,0,0.0458333,0,0,0,0,1,0,0,0,1,0,0,0,0,0.583333,0,0,1,0.0828495,0.431483,1,0,0,0,0,1,0,0.949999,0.05,0,0.949999,0.05,0,0,0.0783808,0.301309,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.15,0.397669,0.386632,0.5103,0.00790743,0.342904,0,0,0,0,1,0,0,0,0,0.399471,0.329051,0.135953,0.268052,0.0412935,0,0.0117647,0.792157,0.682353,0.360784,0.0235294,0,0,0.2,0.196078,0.0196078,0.356863,0.0666667,0.411765,0.0196078,0,0.0862745,0.439216,0.921569,0.101961,0.054902,0.572549,0.129412,0,0.0470588,0.105882,0.282353,0,0,0,0,0.304853,0.474584,0.463961,0.539614,0.524868,0.458878,0.403043,0,0.768627,0.643137,0.337255,0.0627451,0.00784314,0.0352941,0.231373,0.113725,0.0117647,0.313726,0.0392157,0.513726,0.0156863,0.109804,0.909804,0.0980392,0.0745098,0.470588,0.0588235,0,0.0156863,0.0862745,0.592157,0.0705882,0.113725,0.160784,0.0627451,0.200073,0.260522,0,0.906545,0.892134,0.262745,0,0,0.72549,0.52549,0.705882,0.113725,0.4,0.0509804,0.180392,0.741842,0.770078,0.507503,0.341965,0.0170773,0.0356965,0.484645,0,0.0217965,0.355411,0,0,0,0.83397,0.375,0.0483092,0,0,0,0,0,0,0,0,0,0.227053,0,0.515304,0.516081,0,0.530292,0.056287,0.398549,0,0};

            NFSSaveLoad::TFactorsReader baselineReader(
                    baselineDomain.GetBorders(),
                    NFSSaveLoad::CreateRawFloatsInput(baselineFactors.begin(), baselineFactors.size()));
            NFSSaveLoad::TFactorsReader testReader(
                    testDomain.GetBorders(),
                    NFSSaveLoad::CreateRawFloatsInput(testFactors.begin(), testFactors.size()));
            NFactorSlices::TSlicesMetaInfo meta;
            Y_ENSURE(NFactorSlices::NDetail::ReConstructMetaInfo(outputBorders, meta));
            baselineReader.ReadTo(baselineStorage, meta);
            testReader.ReadTo(testStorage, meta);
        }
        TFactorView baselineView = baselineStorage.CreateView();
        TFactorView testView = testStorage.CreateView();

        float* baselineIter = baselineView.GetRawFactors();
        float* testIter = testView.GetRawFactors();
        const TVector<float> baselineFeatures(baselineIter, baselineIter + model.GetNumFeats());
        const TVector<float> testFeatures(testIter, testIter + model.GetNumFeats());

        NMatrixnet::TInfluenceCalcOptions option;
        TVector<float> calcResult = NMatrixnet::GetFactorInfluenceRatings(
            model,
            baselineFeatures, testFeatures, option
        );

        for(auto i : xrange(Min<size_t>(1900, calcResult.size()))) {
            if (calcResult.at(i) != 0) {
                //Cout << calcResult.at(i) << " <- " << i << Endl;
            }
        }
    }
}
}

