#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import print_function
from shutil import copyfile

import yatest.common

NORMALIZE_QUERIES_PATH = yatest.common.binary_path(
    'quality/functionality/entity_search/factqueries/tools/normalize_queries/normalize_queries'
)

SPECIAL_WORDS_GZTPROTO_PATH = yatest.common.source_path(
    'kernel/normalize_by_lemmas/special_words.gztproto'
)

GZT_COMPILER_PATH = yatest.common.binary_path('dict/gazetteer/compiler/gztcompiler')

SPECIAL_WORDS_GZT_CONTENT = '''encoding "utf-8";

import "special_words.gztproto";

TSpecialWordProto m_1 { {"что такое" morph = EXACT_FORM} MARKER }
TSpecialWordProto m_2 { {"что значит" morph = EXACT_FORM} MARKER }
TSpecialWordProto m_3 { {"определение" morph = EXACT_FORM} MARKER }
TSpecialWordProto m_9 { {"что это" morph = EXACT_FORM} MARKER }
TSpecialWordProto m_10 { {"это" morph = EXACT_FORM} MARKER }
TSpecialWordProto m_20 { {"что это такое" morph = EXACT_FORM} MARKER }
TSpecialWordProto m_21 { {"что это значит" morph = EXACT_FORM} MARKER }

TSpecialWordProto m_28 { {"когда" morph = EXACT_FORM} MARKER, Marker="__when__" }

TSpecialWordProto m37 { {"сколько" morph = EXACT_FORM} MARKER, Marker="__how_long__"  }

TSpecialWordProto sw_1 { {"в" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_2 { {"во" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_4 { {"на" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_9 { {"с" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_10 { {"а" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_11 { {"как" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_12 { {"к" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_14 { {"но" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_16 { {"за" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_17 { {"у" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_35 { {"какой" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_39 { {"каком" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_72 { {"ж" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_76 { {"то" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_96 { {"и" morph = EXACT_FORM} STOPWORD }
TSpecialWordProto sw_97 { {"или" morph = EXACT_FORM} STOPWORD }

TSpecialWordProto swe_1 { {"гепатит а " morph = EXACT_FORM} STOPWORD_EXCEPTION }

TSpecialWordProto rw_9 { {"пять" morph = ALL_FORMS} REPLACED_WORD, Replacement="5" }
'''

ALL_PATTERNS_TXT_CONTENT = '''
rule r0 = как пишется слово ..?.? слитно или раздельно;
rule r1 = как пишется слово ..?.?;
rule r2 = как пишется ..?.?;
rule r3 = как пишется ..?.? слитно или раздельно;
rule r4 = как правильно ..?.? или ..?.?;
rule r5 = ..? или ..?;
rule r6 = как правильно написать слово ..?.?;
rule r7 = как правильно написать ..?.?;
rule r8 = как правильно пишется слово ..?.?;
rule r9 = как правильно пишется ..?.?;
'''

BEFORE_AFTER_PAIRS = [
    ('как пишется слово хитрость слитно или раздельно', 'как пишется слово хитрость слитно или раздельно'),
    ('как пишется слово хитрая хитрость слитно или раздельно', 'как пишется слово хитрая хитрость слитно или раздельно'),
    ('как пишется слово хитрая хитрая хитрость слитно или раздельно', 'как пишется слово хитрая хитрая хитрость слитно или раздельно'),
    ('как пишется слово хитрая хитрая хитрая хитрость слитно или раздельно', 'писаться раздельно слитно слово хитрость хитрый'),

    ('приоритезация или приоритизация', 'приоритезация или приоритизация'),

    ('как правильно пишется ёж', 'как правильно пишется ёж'),

    ('как правильно пишется ёж собака медведь кот', 'еж кот медведь писаться правильно собака'),

    ('АЙФОН МАТОВЫЙ', 'айфон матовый'),
    ('amidioteka', 'amidioteka'),
    ('гисметео саратов', 'гисметео саратов'),
    ('капитанша кинопоиск', 'капитанша кинопоиск'),
    ('отель пальмира палас ялта', 'отель палас пальмира ялта'),
    ('окружность головы у детей таблица мальчиков', 'голова мальчик окружность ребенок таблица'),
    ('одноклассники моя страница', 'мой одноклассник страница'),
    ('школьный портал', 'портал школьный'),
    ('расписание электрички путепроводная химки', 'путепроводный расписание химки электричка'),
    ('человек поднял руку', 'подымать рука человек'),

    ('что такое управление в русском языке', 'русский управление язык __marker__'),
    ('что такое девальвация рубля простым языком', 'девальвация простой рубль язык __marker__'),
    ('что такое оксюморон в русском языке определение', 'оксюморон русский язык __marker__'),
    ('идиома это тоже самое что фразеологизм', 'идиома самый тоже фразеологизм что __marker__'),
    ('категория запаса 2 в военном билете что значит', 'билет военный запас категория 2 __marker__'),
    ('брадикардия сердца что это значит у взрослого', 'брадикардия взрослый сердце __marker__'),
    ('пэт кт исследование что это такое',  'исследование кт пэт __marker__'),
    ('что такое детализация', 'детализация __marker__'),
    ('королева-консорт что это', 'консорт королева __marker__'),
    ('что такое ребусы', 'ребусы __marker__'),

    ('скольким городам присвоено звание город-герой', 'герой город звание присваивать скольким'),
    ('всеволожск санкт-петербург расстояние', 'всеволожск петербург расстояние санкт'),
    ('столица израиля иерусалим или тель-авив', 'авив иерусалим израиль столица тель'),
    ('сео-продвижение что это', 'продвижение сео __marker__'),
    ('front-end разработка что это', 'end front разработка __marker__'),
    ('почему-то как пишется', 'писаться почему'),
    ('как пишется почему-то', 'как пишется почему-то'),
    ('уфа-красноусольск расстояние на машине', 'красноусольск машина расстояние уфа'),
    ('административно-правовой метод это', 'административно метод правовой __marker__'),
    ('везет такси санкт-петербург телефон', 'везти петербург санкт такси телефон'),

    ('еж Ёж ёж Еж ёж-ЁЖ ЕЖ-ЕЖ еж-еж ЁЖ-ЁЖ какой это', 'еж __marker__'),

    ('когда день земли', 'день земля __when__'),
    ('когда прививать яблоню весной в каком месяце', 'весна месяц прививать яблоня __when__'),
    ('когда день медика в 2016 в россии', 'день медик россия 2016 __when__'),

    ('во сколько закрываются переходы в метро в москве', 'закрываться метро москва переход __how_long__'),

    ('сколько мне лет', 'год я __how_long__'),

    ('Как приготовить шарлотку', 'приготавливать шарлотка'),

    ('птица, подстреленная князем Гвидоном 6 букв кроссворд', 'буква гвидон князь кроссворд подстреливать птица 6'),
    ('"прикольный" экспромт 8 букв', 'буква прикольный экспромт 8'),
    ('(устар.) искусство сканворд', 'искусство сканворд устар'),
    ('Другое название озера Сиваш кроссворд', 'другой кроссворд название озеро сиваш'),
    (
        'Комнатное растение с большими эффектными листьями, разделёнными на концах на напоминающие рога доли 11 букв кроссворд',
        'большой буква доля комнатный конец кроссворд лист напоминать разделять растение рог эффектный 11'
    ),
    ('роман отечественого писателя Ю. Семенова 12 букв', 'буква отечественый писатель роман семенов ю 12'),
    ('дупель, но иногда и гаршнеп 7 букв', 'буква гаршнеп дупель иногда 7'),
    ('состояние, когда «кошки на душе скребут»', 'душа кошка скрести состояние __when__'),
    ('звезда французкого кино ... Жирардо кроссворд', '... жирардо звезда кино кроссворд французкий'),
    ('ж. трехмачтовое грузовое судно, флейт; есть и полупинки 5 букв кроссворд', 'буква быть грузовой кроссворд полупинка судно трехмачтовый флейта 5'),
    ('пять музыкантов 7 букв', 'буква музыкант 5 7'),
    ('профессия Германа в повести А. Пушкина «Пиковая дама» 7 букв сканворд', 'буква герман дама пиковый повесть профессия пушкин сканворд 7'),
    # TODO(bogolubsky): this quote does not seem OK:
    ('роман Декура "Мудрец ..." 6 букв сканворд', '..." буква декура мудрец роман сканворд 6'),
    ('Польский киноактёр (1927-1967, ''Пепел и алмаз'', ''Крест за отвагу'', ''Йовита'')', 'алмаз йовит киноактер крест отвага пепел польский 1927 1967'),
    # TODO(bogolubsky): this )" does not seem OK:
    ('Тип минерала, относящийся к "Амфиболы (силикаты)" 11 букв кроссворд', ')" амфибол буква кроссворд минерал относиться силикат тип 11'),
    ('I (Ирод Великий) сканворд', 'i великий ирод сканворд'),
    ('Вязкая разновидность каменного угля (поделочный камень)', 'вязкий каменный камень поделочный разновидность уголь'),
    ('Украинская писательница (1907-1982) 6 букв кроссворд', 'буква кроссворд писательница украинский 1907 1982 6'),

    ('телефон +7(123)123-12-12', 'телефон 71231231212'),
    ('телефон +81231231212', 'телефон 8123123121 2'),
    ('телефон +81231231212 84951234567', 'телефон 8123123121 2 84951234567'),
    ('телефон 8(795)1111111', 'телефон 87951111111'),
    ('кто звонил 7-794-111-11-11', 'звонить кто 77941111111'),
    ('+7-794-111-11-11 кто звонил', 'звонить кто 77941111111'),
    ('+7-794-111-11-11 блабла 91231112233', 'блабнуть 77941111111 9123111223 3'),
    ('+7-794-111-11-11 блабла 9-123-111-22-33', 'блабнуть 77941111111 9 123 111 22 33'),
    ('abc 2(999)123-45-45 b la 8999-123-45-45', 'abc b la 2 999 123 45 45 89991234545'),
    ('abc 2(999)123-45-45 b la +7(999)123-45-45', 'abc b la 2 999 123 45 45 79991234545'),
    ('abc 7(999)123 b la 3999-123-45-45', 'abc b la 7 999 123 3999 123 45 45'),

    ('ГепатиТ А', 'а гепатит'),

    ('ดรีมลีกเสื้อบอล', ' ก ด บ ม ร ล ส อ เ')
]


def test_normalize():
    copyfile(SPECIAL_WORDS_GZTPROTO_PATH, 'special_words.gztproto')

    with open('special_words.gzt', 'w') as fout:
        print(SPECIAL_WORDS_GZT_CONTENT, file=fout)

    yatest.common.execute(
        [
            GZT_COMPILER_PATH,
            'special_words.gzt',
            'special_words.gzt.bin'
        ],
        check_exit_code=True
    )

    with open('all_patterns.txt', 'w') as fout:
        print(ALL_PATTERNS_TXT_CONTENT, file=fout)

    with open('queries.tsv', 'w') as fout:
        for pair in BEFORE_AFTER_PAIRS:
            print(pair[0], file=fout)

    yatest.common.execute(
        [
            NORMALIZE_QUERIES_PATH,
            '--gzt', 'special_words.gzt.bin',
            '--regexp', 'all_patterns.txt',
            '--input', 'queries.tsv',
            '--output', 'normalized_queries.tsv'
        ],
        check_exit_code=True
    )

    for line, pair in zip(open('normalized_queries.tsv'), BEFORE_AFTER_PAIRS):
        result = line.strip().split('\t')[1].decode('utf-8')
        expected = pair[1].decode('utf-8')

        assert result == expected
