#pragma once

#include <cstdio>
#include <cstdlib>
#include <google/protobuf/text_format.h>
#include <util/folder/path.h>
#include <util/generic/set.h>
#include <util/generic/yexception.h>
#include <util/generic/string.h>
#include <util/stream/file.h>
#include <util/generic/hash.h>
#include <util/string/cast.h>

class TGenError: public yexception {};

struct TStringStreams: public THashMap<size_t, TAutoPtr<TStringStream>> {
    inline TStringStream& Get(size_t n) {
        auto& ret = (*this)[n];

        if (!ret) {
            ret.Reset(new TStringStream());
        }

        return *ret.Get();
    }
};

struct TCodegenParams {
    TStringStream Hdr;
    TStringStream Cpp;
    TStringStreams ExtraCpp;
    TStringStream Warnings;
    TString HeaderFileName;
    TString TargetSlice;
    const char** ArgvRest = nullptr;
    int ArgcRest = 0;
    size_t CppParts = 0;

    inline TStringStream& CppStream(size_t n) {
        return ExtraCpp.Get(n);
    }

    inline TStringStream& MethodStream(size_t n) {
        if (CppParts) {
            return CppStream(CppParts + n);
        }

        return Cpp;
    }

    inline TStringStream& UniformStream(size_t i, size_t count) {
        if (CppParts) {
            return CppStream(i * CppParts / count);
        }

        return Cpp;
    }
};

template<class TInput>
void PreProcessInputText(TString& inputText, TCodegenParams& params) {
    Y_UNUSED(inputText);
    Y_UNUSED(params);
}

template<class TInput>
int MainImpl(int argc, const char **argv) {
    if (argc < 3) {
        Cerr << "Usage: " << argv[0] << " input.in output.cpp output.h namespace1 namespace2 ..." << Endl;
        return 1;
    }

    TString inputPath = argv[1];
    try {
        TCodegenParams params;

        TString cppPath = argv[2];
        TString headerPath = argv[3];

        int argcRest = argc - 4;
        const char** argvRest = argv + 4;

        // FIXME(mvel): use last_getopt, pass objects count from OUT_NUM

        // read all non-positional args (
        while (argcRest > 0 && TString(argvRest[0]).StartsWith("--")) {
            if (TString(argvRest[0]) == "--cpp-parts") {
                Y_ENSURE(argcRest > 1, "Syntax: --cpp-parts <NUM>. ");
                params.CppParts = FromString<size_t>(argvRest[1]);
                argcRest -= 2;
                argvRest += 2;
                continue;
            }
            if (TString(argvRest[0]) == "--target-slice") {
                Y_ENSURE(argcRest > 1, "Syntax: --target-slice <NUM>. ");
                params.TargetSlice = TString(argvRest[1]);
                argcRest -= 2;
                argvRest += 2;
                continue;
            }
            ythrow yexception() << "Unknown codegen parameter: " << TString(argvRest[0]);
        }

        params.HeaderFileName = TFsPath(headerPath).Basename();
        if (argcRest > 0) {
            params.ArgvRest = argvRest;
            params.ArgcRest = argcRest;
        } else {
            params.ArgvRest = nullptr;
            params.ArgcRest = 0;
        }

        TString inputText = TUnbufferedFileInput(inputPath).ReadAll();
        PreProcessInputText<TInput>(inputText, params);

        TInput input;
        Y_ENSURE(
            google::protobuf::TextFormat::ParseFromString(inputText, &input),
            "Failed to parse file " << inputPath.Quote() << "."
        );

        params.Hdr << "// DO NOT EDIT THIS FILE MANUALLY\n"
                << "// It is automatically generated by code generator from "
                << TFsPath(inputPath).Basename() << " during the build\n\n";

        GenCode(input, params);

        TUnbufferedFileOutput(headerPath).Write(params.Hdr.Str());
        TUnbufferedFileOutput(cppPath).Write(params.Cpp.Str());

        for (auto&& cpp : params.ExtraCpp) {
            const TString dirName = TFsPath(cppPath).Dirname();
            const TString baseName = TFsPath(cppPath).Basename();
            const TString& data = cpp.second->Str();

            TStringBuf fname;
            TStringBuf ext;

            TStringBuf(baseName).Split('.', fname, ext);

            TUnbufferedFileOutput(dirName + "/" + ToString(fname) + "." + ToString(cpp.first) + "." + ToString(ext)).Write(data.data(), data.size());
        }

        if (params.Warnings.Str()) {
            Cerr << "Got some warnings while processing " << inputPath.Quote() << ":" << Endl;
            Cerr << params.Warnings.Str() << Endl;
        }
    } catch (const yexception&) {
        Cerr << "Codegen error in " << inputPath.Quote() << ": " << CurrentExceptionMessage() << Endl;
        return 1;
    }

    return 0;
}
