#include "random_log_parse.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/algorithm.h>

bool ParseRankingData(TString markerEntry, NJson::TJsonValue &dst) {
    TString decoded = Base64Decode(markerEntry);
    TStringInput jsonStream(decoded);
    return NJson::ReadJsonTree(&jsonStream, &dst);
}


Y_UNIT_TEST_SUITE(RandomLogSuite) {
    Y_UNIT_TEST(RandomLogFactorsParse) {
        TString markerEntry = "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";
        NJson::TJsonValue tree;
        UNIT_ASSERT(ParseRankingData(markerEntry, tree));
        TVector<float> factors;
        UNIT_ASSERT(ExtractRankingFactors(tree, factors, NFactorSlices::EFactorSlice::WEB));
        double checksum = std::accumulate(factors.begin(), factors.end(), 0.0);
        UNIT_ASSERT(checksum > 387 && checksum < 388);
    }
    Y_UNIT_TEST(RandomLogFactorsParseFromMarker) {
        TString markerEntry = "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";
        TVector<float> factors;
        UNIT_ASSERT(ExtractRankingFactorsFromMarker(markerEntry, factors, NFactorSlices::EFactorSlice::WEB));
        double checksum = std::accumulate(factors.begin(), factors.end(), 0.0);
        UNIT_ASSERT(checksum > 387 && checksum < 388);
    }

    Y_UNIT_TEST(RandomLogFactorsParseMultipleSlices) {
        TString markerEntry = "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";
        NJson::TJsonValue tree;
        UNIT_ASSERT(ParseRankingData(markerEntry, tree));
        TVector<float> factors, markerFactors;
        UNIT_ASSERT(ExtractRankingFactors(tree, factors, NFactorSlices::EFactorSlice::WEB));
        double checksum = std::accumulate(factors.begin(), factors.end(), 0.0);
        UNIT_ASSERT_DOUBLES_EQUAL(checksum, 507.3685968, std::numeric_limits<float>::epsilon());
        UNIT_ASSERT(ExtractRankingFactors(tree, factors, NFactorSlices::EFactorSlice::ALL));
        checksum = std::accumulate(factors.begin(), factors.end(), 0.0);
        UNIT_ASSERT_DOUBLES_EQUAL(checksum, 660.1468442, std::numeric_limits<float>::epsilon());
    }
    Y_UNIT_TEST(RandomLogHuffFactorsParse) {
        TString markerEntry = "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";
        NJson::TJsonValue tree;
        UNIT_ASSERT(ParseRankingData(markerEntry, tree));
        TVector<float> factors, markerFactors;
        UNIT_ASSERT(ExtractRankingFactors(tree, factors, NFactorSlices::EFactorSlice::WEB));
        double checksum = std::accumulate(factors.begin(), factors.end(), 0.0);
        UNIT_ASSERT(checksum > 307 && checksum < 308);
        UNIT_ASSERT(!ExtractRankingFactors(tree, factors, NFactorSlices::EFactorSlice::ALL));
    }
}
