#include "segment.h"

namespace NSegm {
namespace NPrivate {

void TSegmentStatistics::Add(TDocNode& src, bool inLink) {
    if (IsA<DNT_TEXT> (&src)) {
        Words += src.Props.NWords;

        if (inLink)
            LinkWords += src.Props.NWords;

        TextMarkers |= src.Props.TextMarkers;

        return;
    }

    if (IsA<DNT_INPUT> (&src)) {
        Inputs += src.Props.NInputs;
        return;
    }

    if (IsA<DNT_BREAK> (&src)) {
        Add(src.Props.Level);
        return;
    }

    if (IsA<DNT_LINK> (&src)) {
        Y_VERIFY(!inLink, " ");
        // skip implied links (workaround for html5 parser tree fixing issues)
        if (!src.Props.LinkMarkers.Implied) {
            Links += 1;

            switch (src.Props.LinkType) {
            case LT_EXTERNAL_LINK:
                DomainSet[src.Props.Domain & DomainHashMask] = 1;
                LinkMarkers |= src.Props.LinkMarkers;
                break;
            case LT_SELF_LINK:
                ++SelfLinks;
                [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
            case LT_FRAGMENT_LINK:
                ++FragmentLinks;
                [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
            case LT_LOCAL_LINK:
                ++LocalLinks;
                break;
            }
        }

        for (TDocNode::iterator it = src.Begin(); it != src.End(); ++it)
            this->Add(*it, true);

        return;
    }

    Y_FAIL(" ");
}

TDocNode* TSegmentNode::GetRealFrontBlock() const {
    if (IncludesFrontBlock)
        return FrontBlock;

    TDocNode* b = FrontBlock;
    while(b && b->NodeEnd.SentAligned() <= GetFrontOffset())
        b = b->Parent;

    return b;
}

TDocNode* TSegmentNode::GetRealBackBlock() const {
    if (IncludesBackBlock)
        return BackBlock;

    TDocNode* b = BackBlock;
    while(b && b->NodeStart.SentAligned() >= GetBackOffset())
        b = b->Parent;

    return b;
}

TBlockMarkers TSegmentNode::FrontBlockMarkers() const {
    TDocNode* b = GetRealFrontBlock();
    TBlockMarkers m = TBlockMarkers::New();

    while (b) {
        m |= b->Props.BlockMarkers;
        b = b->Parent;
    }

    return m;
}

TSegmentSpan TSegmentNode::GetZone() const {
    TSegmentSpan z;

    z.Begin = GetFrontOffset();
    z.FirstBlock.Included = IncludesFrontBlock;

    if (TDocNode* f = GetRealFrontBlock()) {
        z.FirstSignature = f->Props.Signature;
        z.FirstTag = f->Props.Tag;
        z.FrontAbsDepth = f->Props.NodeLevel;
    }

    z.LastBlock.Included = IncludesBackBlock;
    z.End = GetBackOffset();

    if (TDocNode* b = GetRealBackBlock()) {
        z.LastSignature = b->Props.Signature;
        z.LastTag = b->Props.Tag;
        z.BackAbsDepth = b->Props.NodeLevel;
    }

    z.CheckedAdd(z.Words, Stats.Words);
    z.CheckedAdd(z.LinkWords, Stats.LinkWords);
    z.CheckedAdd(z.Links, Stats.Links);
    z.CheckedAdd(z.LocalLinks, Stats.LocalLinks);
    z.CheckedAdd(z.FragmentLinks, Stats.FragmentLinks);

    z.HasSelfLink = Stats.SelfLinks;

    z.CheckedAdd(z.Domains, Stats.Domains());
    z.CheckedAdd(z.Inputs, Stats.Inputs);
    z.CheckedAdd(z.Blocks, 1 + Stats.Breaks);

    TBlockMarkers bm = FrontBlockMarkers();
    TTextMarkers fronttm = TTextMarkers::New();

    if (IncludesFrontBlock)
        FrontBlock->MergeTextMarkers(fronttm);

    z.AdsCSS = bm.AdsCSS || Stats.LinkMarkers.AdsCSS;
    z.AdsHref = Stats.LinkMarkers.AdsHref;

    z.CommentsCSS = bm.CommentsCSS;
    z.HasHeader = HasHeader;
    z.AdsHeader = fronttm.AdsText;
    z.CommentsHeader = fronttm.CommentsText;
    z.FooterCSS = bm.FooterCSS;
    z.PollCSS = bm.PollCSS;
    z.MenuCSS = bm.MenuCSS;

    z.FooterText = Stats.TextMarkers.FooterText;
    z.CheckedAdd(z.CommasInText, Stats.TextMarkers.CommasInText);
    z.CheckedAdd(z.SymbolsInText, Stats.TextMarkers.SymbolsInText);
    z.CheckedAdd(z.AlphasInText, Stats.TextMarkers.AlphasInText);
    z.CheckedAdd(z.SpacesInText, Stats.TextMarkers.SpacesInText);

    return z;
}

EBlockStepType GetStep(HT_TAG t) {
    switch (t) {
    default:
        return IsHxTag(t) || HT_P == t || HT_BLOCKQUOTE == t || IsListRootTag(t) ? BST_PARAGRAPH
                : BST_BLOCK;
    case HT_TD:
    case HT_TH:
        return BST_INLINE;
    case HT_CAPTION:
    case HT_DT:
        return BST_SUPITEM;
    case HT_LI:
    case HT_TR:
    case HT_DD:
        return BST_ITEM;
    }
}

void IncrementStep(TBlockInfo& d, TDocNode *& n) {
    Y_VERIFY(n, " ");
    d.Add(GetStep(n->Props.Tag));
    Y_VERIFY((!n->Parent || n->Props.NodeLevel == n->Parent->Props.NodeLevel + 1), " ");

    n = n->Parent;
}

void AddStep(TBlockInfo& d1, TBlockInfo& d2, TDocNode*n1, TDocNode*n2) {
    if (d1.Included && !d2.Included) {
        d1.Add(GetStep(n1->Props.Tag));
        return;
    }

    if (d2.Included && !d1.Included) {
        d2.Add(GetStep(n2->Props.Tag));
        return;
    }
}

TBlockDist SumDist(TBlockInfo& d1, TBlockInfo& d2, TDocNode* n1, TDocNode* n2) {
    Y_VERIFY(n1 && n2, " ");

    bool includesN1 = d1.Included;
    bool includesN2 = d2.Included;
    d1.Depth = 0;
    d2.Depth = 0;

    if (n1 == n2) {
        AddStep(d1, d2, n1, n2);
        return TBlockDist(d1, d2);
    }

    if (!includesN1)
        n1 = n1->Parent;
    if (!includesN2)
        n2 = n2->Parent;

    for (int i = n1->Props.NodeLevel - n2->Props.NodeLevel; i > 0; --i)
        IncrementStep(d1, n1);

    for (int i = n2->Props.NodeLevel - n1->Props.NodeLevel; i > 0; --i)
        IncrementStep(d2, n2);

    while (n1 && n2 && n1 != n2) {
        IncrementStep(d1, n1);
        IncrementStep(d2, n2);
    }

    return TBlockDist(d1, d2);
}

TBlockDist SumDist(TBlockInfo& d1, TBlockInfo& d2, TSegmentNode& prev, TSegmentNode& next) {
    return SumDist(d1, d2, prev.BackBlock, next.FrontBlock);
}

TBlockDist SumDist(TSegmentNode&s1, TSegmentNode&s2) {
    TBlockInfo d1, d2;
    d1.Included = s1.IncludesBackBlock;
    d2.Included = s2.IncludesFrontBlock;
    return SumDist(d1, d2, s1, s2);
}


}
}

