#include "capital.h"
#include <kernel/lemmer/alpha/abc.h>
#include <library/cpp/tokenizer/tokenizer.h>
#include <library/cpp/containers/comptrie/set.h>
#include <library/cpp/containers/comptrie/loader/loader.h>
#include <library/cpp/deprecated/split/split_iterator.h>

namespace {
    // use tools/triecompiler for
    // making set of TUtf16String
    // ./triecompiler -w -0 -m -i capital_blacklist.txt capital_blacklist.trie
    // TUtf16String <-> TUtf16String trie
    // ./triecompiler -w -t TUtf16String -m -i capital_special.txt capital_special.trie

    // use tools/printtrie/ to view:
    // ./printtrie -t TUtf16String capital_special.trie
    // ./printtrie -t 0 capital_blacklist.trie
    const unsigned char CAPITAL_DATA[] = {
        #include <kernel/snippets/simple_textproc/capital/capital.inc>
    };
};

namespace NSnippets {

using TBlackSet = TCompactTrieSet<wchar16>;
using TRulesTrie = TCompactTrie<wchar16, TUtf16String>;
static TBlackSet BLACK_LIST = LoadTrieFromArchive<TBlackSet>("/capital_blacklist.trie", CAPITAL_DATA);
static TRulesTrie SPECIAL_CAPS = LoadTrieFromArchive<TRulesTrie>("/capital_special.trie", CAPITAL_DATA);

static const TUtf16String TURKEY_I_SUFFIX = u"iı";

static bool HasCapitalized(const TWtringBuf& str) {
    for (size_t i = 0; i < str.size(); ++i)
        if (IsUpper(str[i]))
            return true;
    return false;
}

class TUpperHandler: public ITokenHandler {
public:
    TUtf16String Text;
    const NLemmer::TAlphabetWordNormalizer* WordNormalizer;
    TUpperHandler(ELanguage langId)
        : WordNormalizer(NLemmer::GetAlphaRules(langId))
    {}
    void OnToken(const TWideToken& multiToken, size_t /*origleng*/, NLP_TYPE type) override {
        // Капитилизация первого символа слов:
        // - капитализируем слова через '-' (кроме случаев -i и -ı)
        // - не капитализируем слова уже содержащие символы в верхнем регистре
        // - не капитализируем суффиксы после '
        if (!multiToken.Leng)
            return;
        TUtf16String token(multiToken.Text());

        if ((type == NLP_WORD || type == NLP_MARK) && !BLACK_LIST.Find(token)) {
            TUtf16String caps;
            if (SPECIAL_CAPS.Find(token, &caps)) {
                Text.append(caps);
                return;
            }
            bool done = false;
            for (const TCharSpan& span : multiToken.SubTokens) {
                TUtf16String::reference c = token[span.Pos];
                if (span.Pos && span.Len == 1 && TURKEY_I_SUFFIX.Contains(c))
                    continue;
                switch (span.TokenDelim) {
                    case TOKDELIM_APOSTROPHE:
                        done = true;
                        [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
                    case TOKDELIM_MINUS:
                    case TOKDELIM_NULL:
                        if (!HasCapitalized(TWtringBuf(token.begin() + span.Pos, span.Len)))
                            c = WordNormalizer->ToUpper(c);
                        break;
                    default:
                        token = multiToken.Text();
                        done = true;
                        break;
                };
                if (done)
                    break;
            }
        }
        Text.append(token);
    }
};

TUtf16String Capitalize(const TUtf16String& text, ELanguage langId) {
    TUpperHandler upper(langId);
    TNlpTokenizer tokenizer(upper, false);
    tokenizer.Tokenize(text);
    return upper.Text;
}

}
