#include "tokenizer.h"

#include "consts.h"
#include "matcher.h"
#include "tokens.h"
#include "url_wanderer.h"

#include <kernel/snippets/simple_textproc/deyo/deyo.h>

#include <library/cpp/deprecated/dater_old/scanner/scanner.h>
#include <library/cpp/tokenizer/tokenizer.h>
#include <util/generic/list.h>
#include <util/generic/hash_set.h>
#include <util/string/split.h>

namespace NUrlCutter
{
    class TUrlTokenHandler: public ITokenHandler, private TNonCopyable
    {
    private:
        typedef TList<TTokenInfo>::iterator It;

        const TMatcherSearcher& MatcherSearcher;
        const THashSet<TUtf16String>& DynamicStopWords;

        TTokenList&                 Tokens;
        EPathType                   Zone;
        bool                        InDate;

    private:
        void ProcessToken(TUtf16String& token, NLP_TYPE type)
        {
            switch (type) {
                case NLP_MISCTEXT:
                    {
                        TVector<TUtf16String> parts;
                        TContainerConsumer< TVector<TUtf16String> > c(&parts);
                        TSkipEmptyTokens<TContainerConsumer<TVector<TUtf16String> > > sec(&c);
                        TKeepDelimiters<TSkipEmptyTokens<TContainerConsumer<TVector<TUtf16String> > > > kdsec(&sec);
                        static const wchar16 SEPS[] = {PATH_SEP, CGI_SEP, 0};
                        static TSetDelimiter<const wchar16> sd(SEPS);
                        SplitString(const_cast<const wchar16*>(token.begin()), token.end(), sd, kdsec);

                        for (const auto& part: parts) {
                            //на первом / и ? нужно переключать зону, на следующую
                            EPathType ptype = Zone;
                            if (part.size() == 1 && (Zone == PT_DOMAIN || Zone == PT_PATH)) {
                                switch (part[0]) {
                                    case CGI_SEP:
                                        ptype = PT_CGI_SP;
                                        Zone = PT_CGI;
                                        break;
                                    case PATH_SEP:
                                        ptype = PT_PATH_SP;
                                        if (Zone == PT_DOMAIN) {
                                            Zone = PT_PATH;
                                        }
                                        break;
                                }
                            }
                            // no need to normalize this type of token. so use it "as is"
                            TMatcherSearcher::TSearchResult sr = MatcherSearcher.AhoSearch(part);
                            Tokens.push_back(TTokenInfo(part, InDate ? TT_DATE : TT_NONE, ptype, &sr));
                        }
                    }
                    break;
                case NLP_MARK:
                    if (!InDate && Zone != PT_DOMAIN && token.size() > 20) {
                        Tokens.push_back(TTokenInfo(Ellipsis, TT_NONE, Zone));
                        break;
                    }
                    [[fallthrough]]; // AUTOGENERATED_FALLTHROUGH_FIXME
                case NLP_WORD:
                case NLP_INTEGER:
                case NLP_FLOAT:
                    {
                        TUtf16String normalizedToken = to_lower(token);
                        NSnippets::DeSmallYo(normalizedToken);
                        TMatcherSearcher::TSearchResult sr = MatcherSearcher.AhoSearch(normalizedToken);
                        bool prevHM = false;
                        if (!Tokens.empty()) {
                            prevHM = !Tokens.back().Matches.empty();
                            if (!sr.empty() && !prevHM) {
                                Tokens.back().AddRevFreq += 2;
                            }
                        }
                        Tokens.push_back(TTokenInfo(token, InDate ? TT_DATE : TT_WORD, Zone, &sr));
                        if (sr.empty() && prevHM) {
                            Tokens.back().AddRevFreq += 2;
                        }
                    }
                    break;
                default:
                    break;
            }
        }

        inline void Merge(It& it, It& nit)
        {
            const size_t oldLength = it->Token.length();
            it->Token.append(nit->Token);
            it->Len += nit->Len;
            it->AddRevFreq += nit->AddRevFreq;
            THashSet<TWtringBuf> unique;
            for (auto q = it->Matches.begin(); q != it->Matches.end(); ++q) {
                unique.insert(q->Word);
            }
            for (auto q = nit->Matches.begin(); q != nit->Matches.end(); ++q) {
                it->Matches.push_back(*q);
                if (!unique.insert(it->Matches.back().Word).second) {
                    it->Matches.pop_back();
                }
            }
            it->HiliteSpans.reserve(it->HiliteSpans.size() + nit->HiliteSpans.size());
            for (const auto& span : nit->HiliteSpans) {
                it->HiliteSpans.push_back(span);
                it->HiliteSpans.back().Begin += oldLength;
                it->HiliteSpans.back().End += oldLength;
            }
        }

        inline bool IsDates(It& it, It& nit)
        {
            return nit != Tokens.end() && it->TType == TT_DATE && nit->TType == TT_DATE;
        }

        inline bool IsExt(It& name, It& dot, It& ext)
        {
            It tmp = ext;
            SafeInc(tmp);
            if (dot != Tokens.end() && ext != Tokens.end() && (tmp == Tokens.end() || tmp->PType == PT_CGI_SP)) {
                return name->PType == PT_PATH && dot->PType == PT_PATH && ext->PType == PT_PATH &&
                name->TType != TT_NONE && dot->TType == TT_NONE && ext->TType == TT_WORD &&
                ext->Len < 6 && dot->Token == DOT;
            }
            return false;
        }

        inline void SafeInc(It& it)
        {
            if(it != Tokens.end()) {
                ++it;
            }
        }

        inline void ShortPenalty(It& it)
        {
            if (it->TType != TT_WORD || it->PType == PT_DOMAIN) {
                return;
            }
            for (const auto& span : it->HiliteSpans) {
                if (!span.IsStop) {
                    return;
                }
            }
            switch (it->Token.size()) {
                case 1:
                    it->AddRevFreq -= 2;
                    break;
                case 2:
                    it->AddRevFreq -= 1;
                    break;
            }
        }

     public:
        TUrlTokenHandler(TRichTreeWanderer &rTreeWanderer, TTokenList& tokens, ELanguage lang, EPathType ptype)
            : MatcherSearcher(rTreeWanderer.GetMatcherSearcher(lang))
            , DynamicStopWords(rTreeWanderer.GetDynamicStopWords(lang))
            , Tokens(tokens)
            , Zone(ptype)
            , InDate(false)
        {
        }

        void SetInDate(bool value)
        {
            InDate = value;
        }

        void OnToken(const TWideToken& multiToken, size_t /*origleng*/, NLP_TYPE type) override
        {

            if (!multiToken.Leng)
                return;

            TUtf16String mt(multiToken.Token, multiToken.Leng);
            if(multiToken.SubTokens.size() > 1) {
                for (const TCharSpan* it = multiToken.SubTokens.begin(); it != multiToken.SubTokens.end(); ++it) {
                    TUtf16String token(multiToken.Token + it->Pos, it->Len + it->SuffixLen);
                    ProcessToken(token, type);
                    if (it + 1 != multiToken.SubTokens.end()) {
                        TUtf16String hyp(multiToken.Token + it->Pos + it->Len + it->SuffixLen, multiToken.Token + (it + 1)->Pos);
                        ProcessToken(hyp, NLP_MISCTEXT);
                    }
                }
            } else{
                ProcessToken(mt, type);
            }
        }

        void WaveUpdate()
        {
            for (It it = Tokens.begin(); it != Tokens.end(); ++it) {
                it->Relax();
            }
            //process the domain part
            It start = Tokens.begin();
            for (It b = Tokens.begin(); b != Tokens.end() && b->PType == PT_DOMAIN;) {
                It e = b;
                ++e;
                while (e != Tokens.end() && e->PType != PT_PATH_SP && e->PType != PT_CGI_SP)
                    ++e;

                bool oneMoreTry = true;
                do {
                    oneMoreTry = false;
                    It nit = b;
                    It it = nit++;
                    do {
                        while (nit != e && nit->TType == TT_NONE)
                            ++nit;

                        if (nit != e && nit->Update(*it, false, DynamicStopWords)) {
                            oneMoreTry = true;
                            break;
                        }
                        it = nit++;
                    } while (it != e);
                } while (oneMoreTry);

                b = e;
                if (b != Tokens.end())
                    ++b;
                start = b;
            }

            // process last part
            bool wasUpd = true;
            while (wasUpd) {
                wasUpd = false;
                for(It b = start; b != Tokens.end();) {
                    if (b->TType == TT_NONE) {
                        ++b;
                        continue;
                    }

                    It e = b;
                    ++e;
                    bool wasSp = false;
                    while (e != Tokens.end() && e->TType == TT_NONE) {
                        if (e->PType == PT_PATH_SP || e->PType == PT_CGI_SP)
                            wasSp = true;
                        ++e;
                    }
                    if (e == Tokens.end())
                        break;

                    if (e->Update(*b, wasSp, DynamicStopWords))
                        wasUpd = true;
                    b = e;
                }
            }
        }

        void PostProcess()
        {
            WaveUpdate();

            It n2it = Tokens.begin();
            It it = n2it++;
            It nit = n2it++;

            while (it != Tokens.end()) {
                if (IsDates(it, nit)) {
                    Merge(it, nit);
                    n2it = Tokens.erase(nit);
                    nit = n2it++;
                    continue;
                }
                if (IsExt(it, nit, n2it)) {
                    Merge(it, nit);
                    Merge(it, n2it);
                    n2it = Tokens.erase(nit);
                    n2it = Tokens.erase(n2it);
                    nit = n2it++;
                    continue;
                }
                ShortPenalty(it);
                SafeInc(it);
                SafeInc(nit);
                SafeInc(n2it);
            }
        }
    };

    TTokenList TokenizeAndHilite(const TUtf16String& url, TRichTreeWanderer& rTreeWanderer,
                                 ELanguage lang, EPathType ptype) {
        TTokenList tokens;
        TUrlTokenHandler urlTH(rTreeWanderer, tokens, lang, ptype);
        TNlpTokenizer tokenizer(urlTH);
        ptrdiff_t ofs = 0;
        for (const auto& date : NDater::ScanHumanUrl(url.begin(), url.end())) {
            urlTH.SetInDate(false);
            tokenizer.Tokenize(url.begin() + ofs, date.Begin - ofs);
            urlTH.SetInDate(true);
            tokenizer.Tokenize(url.begin() + date.Begin, date.Size());
            ofs = date.End;
        }
        urlTH.SetInDate(false);
        tokenizer.Tokenize(url.begin() + ofs, url.end());
        urlTH.PostProcess();
        return tokens;
    }
}
