#include "switch_hit_codegen.h"
#include <kernel/lingboost/constants.h>
#include <kernel/text_machine/interface/stream_constants.h>

#include <util/generic/hash_set.h>
#include <util/string/join.h>

using namespace NModule;

//   Y_FORCE_INLINE void SwitchHitCodegen(const TMultiHit& hit, EStreamType streamType, const TQueriesHelper& queriesHelper) { [generated by the common codegen]
static const char switchHitPrologue[] =
"        TBlockHitInfo info1{hit.BlockHit};\n"
"        switch (streamType) {\n";
// for every stream used by the machine:
//       case TStream::UsedStreamX: [generated]
static const char switchHitPerStreamPart1[] =
"            AddBlockHit(info1);\n";
// instantiation of AddStreamBlockHit<UsedStreamX> for every unit that supports it
//           if (TGetProc<NCoreParts::TPositionlessTrackerFamily>::IsUnitEnabled()) [generated]
//               TGetProc<NCoreParts::TPositionlessTrackerFamily>::AddStreamBlockHit(info1, TStreamSelector<::NLingBoost::TStream::UsedStreamX>()); [generated]
static const char switchHitPerStreamPart2[] =
"            FinishBlockHit(info1);\n"
"\n"
"            for (size_t j = 0; j != hit.Count; ++j) {\n"
"                auto& wordHit = hit.Hits[j];\n"
"                auto& query = queriesHelper.GetQuery(wordHit.QueryId);\n"
"\n"
"                THitInfo info2{\n"
"                    THit{hit.BlockHit.Position, wordHit, hit.BlockHit.Weight, hit.BlockHit.FormId},\n"
"                    queriesHelper.GetLocalIndex(wordHit.QueryId)};\n"
"                if (Y_UNLIKELY(info2.Hit.Word.QueryId >= queriesHelper.GetMultiQuery().Queries.size())) { Y_ASSERT(false); continue; }\n"
"                if (Y_UNLIKELY(info2.Hit.Word.WordId >= queriesHelper.GetQuery(info2.Hit.Word.QueryId).Words.size())) { Y_ASSERT(false); continue; }\n"
"\n"
"                MACHINE_LOG(\"Core::TMotor\", \"AddHit\", TStringBuilder{}\n"
"                     << \"{Expansion: \" << query.ExpansionType\n"
"                     << \", QueryId: \" << info2.Hit.Word.QueryId\n"
"                     << \", LocalQueryId: \" << queriesHelper.GetLocalIndex(info2.Hit.Word.QueryId)\n"
"                     << \", WordId: \" << info2.Hit.Word.WordId << \"}\");\n"
"\n"
"                switch (query.ExpansionType) {\n";
// for every expansion used by the machine in combination with UsedStreamX:
//               case TExpansion::UsedExpansionY: [generated]
static const char switchHitPerExpansionPart1[] =
//"                    Y_ASSERT(TMotor::IsAllowedExpansion(query.ExpansionType);\n" // we don't have access to TMotor here
"                    AddHit(info2);\n";
// instantiation of AddStreamHit<UsedExpansionY, UsedStreamX> for every unit that supports it
//                   if (TGetProc<...>::IsUnitEnabled()) [generated]
//                       TGetProc<...>::template AddStreamHitBasic<::NLingBoost::TExpansion::UsedExpansionY, TStream::UsedStreamX>(info2); [generated]
static const char switchHitPerExpansionPart2[] =
"                    FinishHit(info2);\n"
"                    break;\n";
static const char switchHitPerStreamPart3[] =
"                default:\n"
"                    MACHINE_LOG(\"Core::TMotor\", \"AddHitIgnoredExpansion\", \"{}\");\n"
"                    break;\n"
"                }\n"
"            }\n"
"            break;\n";
static const char switchHitEpilogue[] =
"        default:\n"
"            MACHINE_LOG(\"Core::TMotor\", \"AddBlockHitIgnoredStream\", \"{}\");\n"
"            break;\n"
"        }\n";
//     } [generated by the common codegen]

// special processing to suppress unused variable/parameter compiler warning
// if there are no streams to process
static const char switchHitStub[] =
"        Y_UNUSED(hit);\n"
"        Y_UNUSED(streamType);\n"
"        Y_UNUSED(queriesHelper);\n"
"        MACHINE_LOG(\"Core::TMotor\", \"AddBlockHitIgnoredStream\", \"{}\");\n";

/*
    We make the following assumptions about machines from Core domain:
    * Supported expansion types can be deduced from the list of units.
    * Supported stream types can be deduced from the list of units
      at one level below.
    * Every unit declares expansion types that it processes
      as specializations for Expansion in
      Core::AddStreamHitBasic<Expansion, Stream>.
      Always-forward implementations of AddStreamHitBasic are not allowed.
    * It is safe to drop word hits with expansion types that are
      not registered in Core::AddStreamHitBasic<Expansion, Stream>.
    * An unit can also process block hits via
      Core::AddStreamBlockHit<Stream>.
      This is the only template that allows always-forward implementation
      (because the list of streams is determined at the level below).
    * Every unit that implements Core::AddStreamHitBasic or
      Core::AddStreamBlockHit has exactly one structural arg;
      this arg is the name of a machine from Tracker domain,
      the "child" machine.
    * Every unit in child machines declares stream types that it processes
      as specializations for StreamType in Tracker::AddStreamBlockHitBasic
      and/or Tracker::AddStreamHitBasic (the first one processes
      block hits, the second one processes word hits).
      Always-forward implementations for both of these are not allowed.
    * It is safe to drop block hits in streams that are not registered
      in Tracker::AddStream[Block]HitBasic in any child machine.
      (Note: Core::TCoreUnit actually processes block hits in all streams
      via non-template AddBlockHit, but the results for unsupported
      streams are never used, so it is safe to skip the processing.)
*/

namespace NTextMachine {
namespace NCore {

    void TSwitchHitCodegen::ProcessMachineDefinition(const NTextMachineParts::TCodeGenInput::TMachineDescriptor& machine)
    {
        if (machine.GetDomain() == "Tracker") {
            TVector<TString> unknownUnits;
            TVector<int>& usedStreams = MachineUsesStreams[machine.GetName()];
            for (size_t unitIndex : xrange(machine.UnitSize())) {
                const TString& unitName = machine.GetUnit(unitIndex).GetName();
                const IUnitInfo* unitInfo = TUnitRegistry::Instance().GetUnitInfo(machine.GetDomain(), unitName);
                if (!unitInfo) {
                    unknownUnits.push_back(unitName);
                    continue;
                }
                for (const char* methodName : {"AddStreamBlockHitBasic", "AddStreamHitBasic"}) {
                    const TUnitMethodInfo* methodInfo = unitInfo->GetMethodInfo(methodName);
                    if (!methodInfo) {
                        continue;
                    }
                    const TMethodTemplateArgInfo* argInfo = methodInfo->GetTemplateArgInfo("StreamType");
                    Y_VERIFY(argInfo, "no argument StreamType in %s::%s()", unitName.data(), methodName);
                    Y_VERIFY(!argInfo->GetAlwaysForward(), "template implementations in Tracker domain are not supported, requested by %s::%s()", unitName.data(), methodName);
                    usedStreams.insert(usedStreams.end(), argInfo->GetInstantiations().begin(), argInfo->GetInstantiations().end());
                }
            }
            Y_VERIFY(unknownUnits.empty(), "Unknown units: %s", JoinSeq(", ", unknownUnits).data());
            SortUnique(usedStreams);
        }
    }

    void TSwitchHitCodegen::Generate(IOutputStream& out, const NTextMachineParts::TCodeGenInput::TMachineDescriptor& machine, NModule::ICodegenHelpers& helpers) const
    {
        TProcessorMethod addStreamBlockHitMethod = TProcessorMethod("AddStreamBlockHit").SetGuarded()
            .AddTypeArg("EStreamType", "Stream")
            .AddArg("const TBlockHitInfo&", "info1")
            .AddArg("TStreamSelector<Stream>", "streamSelector")
            .InstantiateAs(TInstantiateData("AddStreamBlockHit")
                .UseArg("info1")
                .UseArg("TStreamSelector<Stream>()")
            );
        TProcessorMethod addStreamHitMethod = TProcessorMethod("AddStreamHit").SetGuarded()
            .AddTypeArg("EExpansionType", "Expansion")
            .AddTypeArg("EStreamType", "StreamType")
            .AddArg("const THitInfo&", "info2")
            .AddArg("TExpansionSelector<Expansion>", "")
            .AddArg("TStreamSelector<StreamType>", "")
            .InstantiateAs(TInstantiateData("AddStreamHitBasic")
                .UseTypeArg("Expansion")
                .UseTypeArg("StreamType")
                .UseArg("info2")
            );
        THolder<NModule::TProcessorMethodInstantiator> addStreamBlockHit = addStreamBlockHitMethod.CreateExplicitInstantiator();
        THolder<NModule::TProcessorMethodInstantiator> addStreamHit = addStreamHitMethod.CreateExplicitInstantiator();
        Y_ASSERT(addStreamBlockHit && addStreamHit);
        for (size_t unitIndex : xrange(machine.UnitSize())) {
            addStreamBlockHit->RegisterUnit(machine.GetDomain(), unitIndex, machine.GetUnit(unitIndex).GetName());
            addStreamHit->RegisterUnit(machine.GetDomain(), unitIndex, machine.GetUnit(unitIndex).GetName());
        }
        addStreamBlockHit->PrepareInstantiations();
        addStreamHit->PrepareInstantiations();

        TVector<THashSet<int>> unitsUseStreams(machine.UnitSize());
        TSet<int> allUsedStreams; // output is generated in the order of this, want sortedness here
        // ignore all units that support neither AddStreamBlockHit nor AddStreamHit;
        // this includes TCoreUnit without child machines
        // (maybe we will want other purely book-keeping units someday?)
        THashSet<size_t> referencedUnits;
        referencedUnits.insert(addStreamBlockHit->AlwaysForwardUnits().begin(), addStreamBlockHit->AlwaysForwardUnits().end());
        for (const auto& it : addStreamBlockHit->GetInstantiations()) {
            referencedUnits.insert(it.second.begin(), it.second.end());
        }
        Y_VERIFY(!addStreamHit->AlwaysForwardUnits(), "template implementations of Core::AddStreamHit are not supported");
        for (const auto& it : addStreamHit->GetInstantiations()) {
            referencedUnits.insert(it.second.begin(), it.second.end());
        }
        for (size_t unitIndex : referencedUnits) {
            const auto& unit = machine.GetUnit(unitIndex);
            Y_ENSURE(unit.ArgSize() == 1);
            const TVector<int>& streams = MachineUsesStreams.at(unit.GetArg(0).GetName()); // throws if the child is not registered
            unitsUseStreams[unitIndex].insert(streams.begin(), streams.end());
            allUsedStreams.insert(streams.begin(), streams.end());
        }

        if (allUsedStreams.empty()) {
            out << switchHitStub;
            return;
        }

        out << switchHitPrologue;
        for (int stream : allUsedStreams) {
            TString streamStr = "TStream::" + ToString((NLingBoost::EStreamType)stream);
            out << "        case " << streamStr << ":\n";
            out << switchHitPerStreamPart1;
            const TVector<size_t>* units = addStreamBlockHit->GetInstantiations().FindPtr(stream);
            if (!units) {
                units = &addStreamBlockHit->AlwaysForwardUnits();
            }
            for (size_t unitIndex : *units) {
                if (unitsUseStreams[unitIndex].contains(stream)) {
                    const auto& unit = machine.GetUnit(unitIndex);
                    const TString procType = helpers.GetUnitProcType(machine.GetDomain(), unit, unit.ArgSize());
                    out << "            if (" << procType << "::IsUnitEnabled())\n";
                    out << "                " << procType << "::" << addStreamBlockHit->GetCallInstantiated(stream) << ";\n";
                }
            }
            out << switchHitPerStreamPart2;
            for (const auto& it : addStreamHit->GetInstantiations()) {
                TVector<size_t> unitsForExpansion;
                for (size_t unitIndex : it.second) {
                    if (unitsUseStreams[unitIndex].contains(stream)) {
                        unitsForExpansion.push_back(unitIndex);
                    }
                }
                if (unitsForExpansion.empty())
                    continue;
                out << "                case TExpansion::" << ToString((NLingBoost::EExpansionType)it.first) << ":\n";
                out << switchHitPerExpansionPart1;
                for (size_t unitIndex : unitsForExpansion) {
                    const auto& unit = machine.GetUnit(unitIndex);
                    const TString procType = helpers.GetUnitProcType(machine.GetDomain(), unit, unit.ArgSize());
                    TString call = addStreamHit->GetCallInstantiated(it.first);
                    SubstGlobal(call, "StreamType", streamStr);
                    out << "                    if (" << procType << "::IsUnitEnabled())\n";
                    out << "                        " << procType << "::" << call << ";\n";
                }
                out << switchHitPerExpansionPart2;
            }
            out << switchHitPerStreamPart3;
        }
        out << switchHitEpilogue;
    }

}
}
