#include "tm_codegen.h"

#include <kernel/text_machine/parts/common/interfaces.h>

#include <library/cpp/getopt/last_getopt.h>
#include <util/folder/path.h>

int main_exc(int argc, const char **argv) {
    TVector<TString> namespaces;
    TString outputPath;

    NLastGetopt::TOpts opts;
    opts.AddLongOption('n', "namespace", "Generate the code in this namespace; multiple options = nested namespaces")
        .RequiredArgument("STRING")
        .AppendTo(&namespaces);
    opts.AddLongOption('o', "output-path", "Path for output files")
        .RequiredArgument("DIR")
        .DefaultValue(".")
        .StoreResult(&outputPath);
    opts.SetFreeArgsMin(1);
    opts.SetFreeArgDefaultTitle("FILE", "machine description as prototext");

    NLastGetopt::TOptsParseResult cmdLine(&opts, argc, argv);
    TVector<TString> inputFileNames = cmdLine.GetFreeArgs();
    TVector<NTextMachineParts::TCodeGenInput> inputs(inputFileNames.size());

    TTextMachinePartsCodegen<::NTextMachine::NCore::TStaticTextMachineInterfaces> codegen;

    for (size_t pos : xrange(inputFileNames.size())) {
        TString inputText = TUnbufferedFileInput(inputFileNames[pos]).ReadAll();
        Y_ENSURE(google::protobuf::TextFormat::ParseFromString(inputText, &inputs[pos]),
            "Failed to parse file " << inputFileNames[pos].Quote() << ".");
        codegen.LoadDefinitions(inputs[pos]);
    }

    codegen.SetupNamespaces(namespaces);

    for (size_t pos : xrange(inputs.size())) {
        TCodegenParams params;
        TString inputBaseName = TFsPath(inputFileNames[pos]).Basename();
        TStringBuf inputNoExtension = TStringBuf(inputBaseName).RBefore('.');
        TString cppPath = JoinFsPaths(outputPath, TString::Join(inputNoExtension, ".cpp"));
        params.HeaderFileName = TString::Join(inputNoExtension, ".h");
        TString headerPath = JoinFsPaths(outputPath, params.HeaderFileName);
        params.Hdr << "// DO NOT EDIT THIS FILE MANUALLY\n"
            << "// It is automatically generated by code generator from "
            << inputBaseName << " during the build\n\n";

        codegen.GenCode(inputs[pos], params);

        TUnbufferedFileOutput(headerPath).Write(params.Hdr.Str());
        TUnbufferedFileOutput(cppPath).Write(params.Cpp.Str());
    }
    return 0;
}

int main(int argc, const char **argv) {
    try {
        return main_exc(argc, argv);
    } catch (const yexception& e) {
        Cerr << "Codegen error: " << e.what() << '\n';
        return 1;
    }
}
