#pragma once

#include "utils.h"
#include <kernel/ugc/aggregation/proto/review_meta.pb.h>

#include <library/cpp/scheme/scheme.h>

#include <util/datetime/base.h>
#include <util/generic/hash.h>
#include <util/generic/maybe.h>
#include <util/generic/ptr.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/generic/list.h>
#include <util/string/builder.h>

namespace NUgc {
    class TFeedbackReader {
    public:

        class IUpdateIdBuilder {
        public:
            virtual ~IUpdateIdBuilder() = default;

            virtual void OnFeedback(const NSc::TValue& feedback) = 0;

            virtual TString Result() = 0;
        };

        struct TRecord {
            TInstant Time;
            TString Value;
            bool IsGeneratedByRating = false; // Set if feedbackId generated by rating update in FeedbackReader
        };

        using TErrorDescr = NFeedbackReaderUtils::TErrorDescr;
        using TAppendOptions = NFeedbackReaderUtils::TAppendOptions;
        using TObjectProps = THashMap<TString, TRecord>;
        using TTable = THashMap<TString, TObjectProps>;
        using TReviewsMap = TMap<TInstant, TString>; // TMap<reviewTime, reviewText>

    public:
        TFeedbackReader();

        TFeedbackReader(TStringBuf tableKeyName);

        void SetUpdateIdBuilder(THolder<IUpdateIdBuilder> builder);

        bool AppendFeedback(const NSc::TValue& feedback, TErrorDescr& error, const TAppendOptions& options = TAppendOptions());

        template <typename Container>
        size_t AppendFeedbacks(const Container& container, TErrorDescr& error, const TAppendOptions& options = TAppendOptions()) {
            size_t counter = 0;

            for (const NSc::TValue& feedback : container) {
                TFeedbackReader::TErrorDescr feedbackError;
                if (!AppendFeedback(feedback, feedbackError, options)) {
                    error.PushUpdate(std::move(feedbackError));
                } else {
                    ++counter;
                }
            }

            return counter;
        }

        size_t LoadFromUserData(const NSc::TValue& userData, TFeedbackReader::TErrorDescr& error, const TAppendOptions& options = TAppendOptions());

        TInstant GetObjectReviewTime(TStringBuf objectName) const;
        TInstant GetObjectRatingTime(TStringBuf objectName) const;

        TMaybe<TString> GetReviewByTimestamp(TStringBuf objectName, const TInstant& t) const;

        TMaybe<TString> GetFeedbackIdByTimestamp(TStringBuf objectName, const TInstant& t) const;

        TReviewMeta GetObjectReviewMeta(TStringBuf objectName) const;

        const TVector<TString>& GetObjectPhotos(TStringBuf objectName) const;

        TMaybe<TStringBuf> Value(TStringBuf table, TStringBuf object, TStringBuf column) const;

        const TObjectProps* GetObjectProps(TStringBuf table, TStringBuf object) const;

        NSc::TValue Aggregate(bool keepEmptyTombstone, bool sortEveryTable = false) const;

        NSc::TValue Aggregate() const {
            return Aggregate(false);
        }

    private:
        bool CheckUserId(const NSc::TValue& feedback);

        bool FeedbackConstraints(const NSc::TValue& feedback, TErrorDescr& error);

    private:
        TString UserId_;
        TString AppId_;
        TString TableKeyName_;

        TMaybe<TInstant> StartAt_;
        TMaybe<TInstant> FinishAt_;

        THolder<IUpdateIdBuilder> UpdateIdBuilder_;

        THashMap<TString, TTable> Data_;
        THashMap<TString, TReviewsMap> ReviewsData_;
        THashMap<TString, TMap<TInstant, TString>> RatingsData_;
        THashMap<TString, TMap<TInstant, TString>> FeedbackIdData_;
        THashMap<TString, TVector<TString>> PhotosData_;
        THashMap<TString, TMap<TInstant, TReviewMeta>> ReviewsMeta_;
    };

    TString BuildUserReviewFeedbackId(const TString& userId, const TString& parentId, const TString& contextId);
}
