#pragma once

#include <util/generic/fwd.h>
#include <util/system/defaults.h>

enum PARSED_TYPE {
    PARSED_ERROR = -2,
    PARSED_EOF = -1,
    PARSED_TEXT,  // "-" (CHUNK_TEXT)
    PARSED_MARKUP // comments, pi, declarations, unused or empty tags
};

enum BREAK_TYPE {
    BREAK_NONE,      // CHUNK_NOBR
    BREAK_DOMPATH,   // dom path changed at this point
    BREAK_WORD,      // CHUNK_EOW (NEW_WORD)
    BREAK_SENTENCE,  // CHUNK_EOS (NEW_SENTENCE)
    BREAK_PARAGRAPH, // CHUNK_EOP (NEW_PARAGRAPH)
    BREAK_CHAPTER,   // CHUNK_EOC (NEW_CHAPTER)
    BREAK_BODY,      // ????????? (hiliting)
    BREAK_DOCUMENT   // !=EOF     (NEW_DOCUMENT) (for multi-doc files)
};

enum SPACE_MODE {
    SPACE_DEFAULT,  // default, don't look for para-breaks COLLAPSE
    SPACE_PRESERVE, // <pre> <listing> <xmp> <plaintext> <td nowrap>
    SPACE_NOBREAK   // <nobr>
};

enum TEXT_WEIGHT {
    WEIGHT_ZERO = -1, // NOINDEX_RELEV
    WEIGHT_LOW,       // LOW_RELEV
    WEIGHT_NORMAL,    // NORMAL_RELEV
    WEIGHT_HIGH,      // HIGH_RELEV   (H1,H2,H3,ADDRESS,CAPTION)
    WEIGHT_BEST       // BEST_RELEV   (TITLE)
};

enum ATTR_POS {
    APOS_DOCUMENT, // #ATTR="VALUE"      [docID].[000000].[000000|wordID]
    APOS_ZONE      // (ZONE?) #ATTR="VALUE"  [docID].[paraID].[wordID]
};

enum ATTR_TYPE {
    ATTR_LITERAL,      // store as single word (#CHARSET='KOI8-R')
    ATTR_URL,          // #URL='http://www.comptek.ru/index.html'
    ATTR_DATE,         // #DATE=1998-09-04  #TIME=1998-09-04T11:53:00Z (ISO-8601)
    ATTR_INTEGER,      // #SIZE=00000002467 ("%011u" decimal number)
    ATTR_BOOLEAN,      // #POETRY=1 (yes/no;1/0;true/false;on/off)
    ATTR_UNICODE_URL,  // URL converted to UTF8 and percent-encoded
    ATTR_NOFOLLOW_URL, // URL in nofollow links
};

enum MARKUP_TYPE {
    MARKUP_NORMAL,  // ok
    MARKUP_IMPLIED, // parser adds
    MARKUP_IGNORED  // parser ignores
};

const TString& ToString(PARSED_TYPE);
bool FromString(const TString& name, PARSED_TYPE& ret);
const TString& ToString(BREAK_TYPE);
bool FromString(const TString& name, BREAK_TYPE& ret);
const TString& ToString(SPACE_MODE);
bool FromString(const TString& name, SPACE_MODE& ret);
const TString& ToString(TEXT_WEIGHT);
bool FromString(const TString& name, TEXT_WEIGHT& ret);
const TString& ToString(ATTR_POS);
bool FromString(const TString& name, ATTR_POS& ret);
const TString& ToString(ATTR_TYPE);
bool FromString(const TString& name, ATTR_TYPE& ret);
const TString& ToString(MARKUP_TYPE);
bool FromString(const TString& name, MARKUP_TYPE& ret);

// PP stands for Parser Property
// N.B. about PP_BASE property.
// Base url must be a valid absolute http url in the sense of THttpURL.
// It means it should satisfy url.Parse(p, THttpURL::FeaturesRobot) == THttpURL::ParsedOK && url.IsValidAbs().
// Otherwise, all linkint events will get ignored.
const char* const PP_BASE = "base";         // base url set by client, pivot of locality calculation
const char* const PP_AUTOBASE = "autobase"; // base url from document
const char* const PP_CHARSET = "charset";   // reliable charset set by client
const char* const PP_DEFCHARSET = "defcharset";
const char* const PP_LANGUAGE = "language";
const char* const PP_DEFLANGUAGE = "deflanguage";
const char* const PP_FRAME = "frame";
const char* const PP_LINKS_HASH = "linkshash";
const char* const PP_ROBOTS = "robots";
const char* const PP_YANDEX = "yandex";
const char* const PP_REFRESH = "refresh";
const char* const PP_REFRESHTIME = "refreshtime";
const char* const PP_LOCALFILE = "localfile";
const char* const PP_INDEXRES = "indexresult";
const char* const PP_ALTERNATE = "alternate";

const char* const PP_IGNORE_PUNCT_BREAKS = "IGNORE_PUNCT_BREAKS"; // ignore sentence and para breaks, generated by internal lexer

const char* const ZONE_EMPTY = "empty";
const char* const ZONE_DOC = "doc";
const char* const ZONE_ANY = "any";
const char* const ZONE_WILD = "_";
const char* const ZONE_SUPER_WILD = "__";

struct PARSED_FLAGS {
    /*BREAK_TYPE */ i8 brk;    /* 3/2 (none,word,para,body) */
    /*SPACE_MODE */ i8 space;  /* 2/1 (collapse,preserve) */
    /*ATTR_TYPE  */ i8 atype;  /* 5/3 (text,literal,url,date,number) */
    /*ATTR_POS   */ i8 apos;   /* 3/1 (document,zone) */
    /*TEXT_WEIGHT*/ i8 weight; /* 3/2 (low,normal,high,best) */
    /*MARKUP_TYPE*/ i8 markup; /* 3/1 (expl,impl) */
    /*PARSED_TYPE*/ i16 type;  /* 13/3 (err,eof,text,attr,open,close,markup) */
    PARSED_FLAGS()
        : brk(BREAK_NONE)
        , space(SPACE_DEFAULT)
        , atype(ATTR_LITERAL)
        , apos(APOS_DOCUMENT)
        , weight(WEIGHT_NORMAL)
        , markup(MARKUP_NORMAL)
        , type(PARSED_TEXT)
    {
    }
};
