package solomon

import (
	"testing"

	dto "github.com/prometheus/client_model/go"
	"github.com/stretchr/testify/assert"
	"go.uber.org/atomic"

	"a.yandex-team.ru/library/go/ptr"
)

func TestPrometheusMetrics(t *testing.T) {
	testCases := []struct {
		name      string
		metrics   []*dto.MetricFamily
		expect    *Metrics
		expectErr error
	}{
		{
			name: "success",
			metrics: []*dto.MetricFamily{
				{
					Name: ptr.String("subregister1_mygauge"),
					Help: ptr.String(""),
					Type: func(mt dto.MetricType) *dto.MetricType { return &mt }(dto.MetricType_GAUGE),
					Metric: []*dto.Metric{
						{
							Label: []*dto.LabelPair{
								{Name: ptr.String("ololo"), Value: ptr.String("trololo")},
							},
							Gauge: &dto.Gauge{Value: ptr.Float64(42)},
						},
					},
				},
				{
					Name: ptr.String("subregisters_count"),
					Help: ptr.String(""),
					Type: func(mt dto.MetricType) *dto.MetricType { return &mt }(dto.MetricType_COUNTER),
					Metric: []*dto.Metric{
						{
							Label:   []*dto.LabelPair{},
							Counter: &dto.Counter{Value: ptr.Float64(2)},
						},
					},
				},
				{
					Name: ptr.String("subregister1_subregister2_myhistogram"),
					Help: ptr.String(""),
					Type: func(mt dto.MetricType) *dto.MetricType { return &mt }(dto.MetricType_HISTOGRAM),
					Metric: []*dto.Metric{
						{
							Label: []*dto.LabelPair{
								{Name: ptr.String("ololo"), Value: ptr.String("trololo")},
								{Name: ptr.String("shimba"), Value: ptr.String("boomba")},
							},
							Histogram: &dto.Histogram{
								SampleCount: ptr.Uint64(4),
								SampleSum:   ptr.Float64(4.2),
								Bucket: []*dto.Bucket{
									{CumulativeCount: ptr.Uint64(1), UpperBound: ptr.Float64(1)}, // 0.5 written
									{CumulativeCount: ptr.Uint64(3), UpperBound: ptr.Float64(2)}, // 1.5 & 1.7 written
									{CumulativeCount: ptr.Uint64(4), UpperBound: ptr.Float64(3)}, // 2.2 written
								},
							},
						},
					},
				},
			},
			expect: &Metrics{
				metrics: []Metric{
					&Gauge{
						name:  "subregister1_mygauge",
						tags:  map[string]string{"ololo": "trololo"},
						value: *atomic.NewFloat64(42),
					},
					&Counter{
						name:  "subregisters_count",
						tags:  map[string]string{},
						value: *atomic.NewInt64(2),
					},
					&Histogram{
						name:         "subregister1_subregister2_myhistogram",
						tags:         map[string]string{"ololo": "trololo", "shimba": "boomba"},
						bucketBounds: []float64{1, 2, 3},
						bucketValues: []int64{1, 2, 1},
					},
				},
			},
			expectErr: nil,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			s, err := PrometheusMetrics(tc.metrics)

			if tc.expectErr == nil {
				assert.NoError(t, err)
			} else {
				assert.EqualError(t, err, tc.expectErr.Error())
			}

			assert.Equal(t, tc.expect, s)
		})
	}
}
