package protonaming

import (
	"go/ast"

	"golang.org/x/tools/go/analysis"
)

const (
	Name = "protonaming"

	protocGenComment = "// Code generated by protoc-gen-go. DO NOT EDIT."
)

var forbiddenPackageNames = map[string]struct{}{
	"proto":  {},
	"public": {},
}

var Analyzer = &analysis.Analyzer{
	Name: Name,
	Doc:  `check naming of protobuf packages`,
	Run:  run,
}

func isProtoPackage(f *ast.File) bool {
	for _, comment := range f.Comments {
		for _, line := range comment.List {
			if line.Text == protocGenComment {
				return true
			}
		}
	}

	return false
}

func run(pass *analysis.Pass) (interface{}, error) {
	for _, f := range pass.Files {
		if isProtoPackage(f) {
			name := pass.Pkg.Name()
			if _, banned := forbiddenPackageNames[name]; banned {
				pass.Reportf(f.Package, "don't give go package a generic name")
				pass.Reportf(f.Package, "rename go package to more specific: option go_package = \"%s;myapi\"", pass.Pkg.Path())
				break
			}
		}
	}

	return nil, nil
}
