#!/bin/bash

set -x -euo pipefail
DAEMON_NAME=loadtest-base-packer

log_fatal (){
    logger -ip daemon.error -t ${DAEMON_NAME} "$@"
    exit 1
}

log_info (){
    logger -ip daemon.info -t ${DAEMON_NAME} "$@"
}

# Directory for temporary files
TMPDIR=$(mktemp -d)

system_upgrade() {
    DEBIAN_FRONTEND=noninteractive apt-get update -qq || log_fatal "Can't update repos"
    DEBIAN_FRONTEND=noninteractive LANG=C apt-get purge -y unattended-upgrades || log_fatal "Can't purge unattended-upgrades"
    DEBIAN_FRONTEND=noninteractive LANG=C apt-get install -y locales || log_fatal "Can't install locales"
    locale-gen en_US.UTF-8 || log_fatal "Failed to set locales"
    dpkg-reconfigure --frontend=noninteractive locales || log_fatal "Can't rebuild locales"
}

install_utils() {
    DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends -yq htop \
        postgresql-client \
	apparmor-ycloud-ycommon-prof \
	apparmor-ycloud-svc-loadtesting-server-prof \
        screen || log_fatal "Can't install utils"
}

remove_unused() {
    DEBIAN_FRONTEND=noninteractive apt-get remove -yq salt-common \
        salt-minion
}

clean_apt() {
    DEBIAN_FRONTEND=noninteractive apt-get clean
}

get_goss() {
    GOSS_BIN="https://github.com/aelsabbahy/goss/releases/download/v0.3.9/goss-linux-amd64"
    curl -fS -L --connect-time 3 --max-time 30 --retry 10 "${GOSS_BIN}" -o ${TMPDIR}/goss || log_fatal "Cant download goss"
    chmod +x ${TMPDIR}/goss
    cp ${TMPDIR}/goss /usr/local/bin/goss || log_fatal "Cant install goss"
    ln -s /usr/local/bin/goss /usr/bin/goss
}

goss_fixes() {
    # disable sshd for ipv4
    echo "AddressFamily inet6" | tee -a /etc/ssh/sshd_config
    # sysctl
    sed -i -e s/1/2/g /etc/sysctl.d/10-ptrace.conf
    sed -i -e s/net.ipv4.tcp_max_syn_backlog\ =\ .*/net.ipv4.tcp_max_syn_backlog\ =\ 4096/g /etc/sysctl.d/30-network.conf
    echo "kernel.unprivileged_bpf_disabled = 1" | tee -a /etc/sysctl.d/10-unprivileged_bpf.conf
    echo "kernel.dmesg_restrict = 1" | tee -a /etc/sysctl.d/99-sysctl.conf
    echo "kernel.seccomp.actions_logged = kill_process kill_thread trap errno trace log" | tee -a /etc/sysctl.d/12-seccomp.conf
    echo "kernel.seccomp.actions_avail = kill_process kill_thread trap errno trace log allow" | tee -a /etc/sysctl.d/12-seccomp.conf
    echo "kernel.perf_event_paranoid = 2" | tee -a /etc/sysctl.d/10-perf_event.conf
    # chmode
    chown root:root /etc/shadow
    chmod 0640 /etc/group
    chmod 0640 /etc/passwd
    echo "install cramfs /bin/true" | tee -a /etc/modprobe.d/CIS.conf
    echo "install freevxfs /bin/true"| tee -a /etc/modprobe.d/CIS.conf
    echo "install jffs2 /bin/true"| tee -a /etc/modprobe.d/CIS.conf
    echo "install hfs /bin/true"| tee -a /etc/modprobe.d/CIS.conf
    echo "install hfsplus /bin/true"| tee -a /etc/modprobe.d/CIS.conf
    echo "install udf /bin/true"| tee -a /etc/modprobe.d/CIS.conf
    echo "blacklist usb-storage" | tee -a /etc/modprobe.d/blacklist.conf
    update-initramfs -u
}

cleanup () {
    rm -rf ${TMPDIR}
    rm -rf /var/log/apt/*
    rm -rf /var/log/{auth,dpkg,syslog,wtmp,kern,alternatives,auth,btmp,dmesg}.log
    rm -rf /var/lib/dhcp/*

}
main() {
    system_upgrade
    install_utils
    get_goss
    remove_unused
    clean_apt
    cleanup
#   goss_fixes
}

main
