#!/bin/bash

set -x -euo pipefail
DAEMON_NAME=cloud-yandex-tank

log_fatal (){
    logger -ip daemon.error -t ${DAEMON_NAME} "$@"
    exit 1
}

log_info (){
    logger -ip daemon.info -t ${DAEMON_NAME} "$@"
}

# Directory for temporary files
TMPDIR=$(mktemp -d)

system_upgrade() {
    DEBIAN_FRONTEND=noninteractive apt-get update -qq || log_fatal "Can't update repos"
    DEBIAN_FRONTEND=noninteractive LANG=C apt-get purge -y unattended-upgrades || log_fatal "Can't purge unattended-upgrades"
    DEBIAN_FRONTEND=noninteractive LANG=C apt-get install -y locales || log_fatal "Can't install locales"
    locale-gen en_US.UTF-8 || log_fatal "Failed to set locales"
    dpkg-reconfigure --frontend=noninteractive locales || log_fatal "Can't rebuild locales"
    echo "* hard nofile 65536" >> /etc/security/limits.d/custom.conf
    echo "* soft nofile 65536" >> /etc/security/limits.d/custom.conf
}

install_utils() {
    DEBIAN_FRONTEND=noninteractive apt-get install --no-install-recommends -yq net-tools \
        python3.9 \
        python3-pip \
        python3-venv \
        jq || log_fatal "Can't install utils" 
}

install_grpcurl() {
    GRPCURL_ARCHIVE="https://github.com/fullstorydev/grpcurl/releases/download/v1.8.2/grpcurl_1.8.2_linux_x86_64.tar.gz"
    curl -fS -L --connect-time 3 --max-time 30 --retry 10 "${GRPCURL_ARCHIVE}" -o ${TMPDIR}/grpcurl.tar.gz || log_fatal "Can't download grpcurl"
    pushd ${TMPDIR}
    tar -zxvf grpcurl.tar.gz || log_fatal "Can't extract grpcurl"
    cp ${TMPDIR}/grpcurl /usr/local/bin/grpcurl || log_fatal "Can't install grpcurl"
    popd
    ln -s /usr/local/bin/grpcurl /usr/bin/grpcurl
}

install_phantom() {
    add-apt-repository ppa:yandex-load/main -y && \
    apt-get update -q && \
    apt-get install -yq \
        phantom=0.14.0~pre65load5nmu \
        phantom-ssl=0.14.0~pre65load5nmu
}

install_pandora() {
    PANDORA_BINARY="https://github.com/yandex/pandora/releases/download/v0.3.8/pandora_0.3.8_linux_amd64"
    curl -fS -L --connect-time 3 --max-time 30 --retry 10 "${PANDORA_BINARY}" -o ${TMPDIR}/pandora || log_fatal "Can't download pandora"
    chmod +x ${TMPDIR}/pandora
    cp ${TMPDIR}/pandora /usr/local/bin/pandora || log_fatal "Can't install pandora"
    mkdir -p /usr/share/doc/pandora
}

install_telegraf() {
    wget --progress=dot:giga https://dl.influxdata.com/telegraf/releases/telegraf_1.4.3-1_amd64.deb
    dpkg -i telegraf_1.4.3-1_amd64.deb
    rm -f telegraf_1.4.3-1_amd64.deb*
}

highload_twick() {
    echo "* hard nofile 65536" >> /etc/security/limits.conf
    echo "* soft nofile 65536" >> /etc/security/limits.conf
    echo "root hard nofile 65536" >> /etc/security/limits.conf
    echo "root soft nofile 65536" >> /etc/security/limits.conf
    echo "net.core.netdev_max_backlog=10000" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.core.somaxconn=262144" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.ipv4.tcp_max_tw_buckets=720000" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.ipv4.tcp_timestamps=1" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.ipv4.tcp_tw_reuse=1" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.ipv4.tcp_fin_timeout=30" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.ipv4.tcp_keepalive_time=1800" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.core.rmem_max=26214400" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.core.rmem_default=6250000" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.core.wmem_max=26214400" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.core.wmem_default=6250000" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.ipv4.tcp_rmem='4096 6250000 26214400'" >> /etc/sysctl.d/12-highload-network.conf
    echo "net.ipv4.tcp_wmem='4096 6250000 26214400'" >> /etc/sysctl.d/12-highload-network.conf
}

add_user() {
    adduser --system --shell /bin/bash --gecos 'Loadtesting user' --group --disabled-password --home /home/loaduser loaduser
}

clean_apt() {
    DEBIAN_FRONTEND=noninteractive apt-get clean
}

cleanup () {
    rm -rf /var/log/apt/*
    rm -rf /var/log/{auth,dpkg,syslog,wtmp,kern,alternatives,auth,btmp,dmesg}.log
    rm -rf /var/lib/dhcp/*
    rm -rf /etc/apt/sources.list.d/yandex-load-ubuntu-main-focal.list
    rm -rf /etc/apt/trusted.gpg.d/yandex-load_ubuntu_main.gpg
    rm -rf /etc/network/interfaces.d/*
    truncate -s 0 /home/ubuntu/.ssh/authorized_keys || log_fatal "Failed to clean authorized_keys"
    truncate -s 0 /root/.ssh/authorized_keys || log_fatal "Failed to clean root authorized_keys"
    rm -rf /var/log/cloud-init* || log_fatal "Failed to remove cloud-init logs"
    rm -f /etc/hostname
}

main() {
    system_upgrade
    install_utils
    install_grpcurl
    install_pandora
    install_telegraf
    highload_twick
    install_phantom
    add_user
    clean_apt
    cleanup
}

main
