package main

import (
	"os"
	"time"

	"github.com/yandex/pandora/cli"
	"github.com/yandex/pandora/core"
	"github.com/yandex/pandora/core/aggregator/netsample"
	coreimport "github.com/yandex/pandora/core/import"
	"github.com/yandex/pandora/core/register"
)

var (
	outlog   *os.File
	shootime int64 
)

type (
	Ammo struct {
		Tag  string
		Text string
	}

	GunConfig struct {
		Target string `validate:"required"`
	}

	Sample struct {
		Tag              string
		ShootTimeSeconds float64
		Code             int
	}

	WriteGun struct {
		// Configured on construction.
		conf GunConfig
		// Configured on Bind, before shooting
		aggr core.Aggregator
		core.GunDeps
	}

	ReadGun struct {
		// Configured on construction.
		conf GunConfig
		// Configured on Bind, before shooting
		aggr core.Aggregator
		core.GunDeps
	}
)

func NewReadGun(conf GunConfig) *ReadGun {
	return &ReadGun{conf: conf}
}

func NewWriteGun(conf GunConfig) *WriteGun {
	return &WriteGun{conf: conf}
}

func (g *ReadGun) Bind(aggr core.Aggregator, deps core.GunDeps) error {
	g.aggr = aggr
	g.GunDeps = deps
	return nil
}

func (g *WriteGun) Bind(aggr core.Aggregator, deps core.GunDeps) error {
	g.aggr = aggr
	g.GunDeps = deps
	return nil
}

func (g *ReadGun) Shoot(ammo core.Ammo) {
	customAmmo := ammo.(*Ammo)
	go g.readWS(customAmmo)
}

func (g *WriteGun) Shoot(ammo core.Ammo) {
	customAmmo := ammo.(*Ammo)
	g.writeWS(customAmmo)
}

func (g *ReadGun) reportDuration(tag string, t time.Duration, code int) {
	sample := netsample.Acquire(tag)
	sample.SetTimestamp(t, code)
	g.aggr.Report(sample)
}

func main() {
	outlog, _ = os.OpenFile("out.log", os.O_CREATE|os.O_WRONLY, 0666)
	// Standard imports.
	fs := coreimport.GetFs()
	coreimport.Import(fs)
	// Custom imports. Integrate your custom types into configuration system.
	coreimport.RegisterCustomJSONProvider("websocket_provider", func() core.Ammo { return &Ammo{} })

	register.Gun("writews", NewWriteGun, func() GunConfig {
		return GunConfig{
			Target: "default target",
		}
	})

	register.Gun("readws", NewReadGun, func() GunConfig {
		return GunConfig{
			Target: "default target",
		}
	})

	cli.Run()
}