#pragma once

#include <yamail/data/serialization/json_writer.h>
#include <mail/akita/service/include/account/featured_account.h>
#include <mail/akita/service/include/blackbox/blackbox.h>
#include <boost/lambda/construct.hpp>
#include <boost/functional.hpp>
#include <boost/bind/bind.hpp>

namespace akita {
namespace reflection {

struct AuthResultStruct
{
    struct DisplayNameStruct
    {
        DisplayNameStruct(const akita::DisplayName& dn) :
                name(dn.name),
                social(static_cast<bool>(dn.socialInfo)),
                socialInfo(dn.socialInfo.value_or(SocialInfo())),
                avatar(dn.avatar) {
        }
        std::string name;
        bool social;
        SocialInfo socialInfo;
        std::string avatar;
    };

    struct AddressRepositoryStruct
    {
        AddressRepositoryStruct(const akita::FeaturedAccount& repo) {
            internalAddresses = repo.internal;
            defaultAddress = repo.defaultAddress;
            validatedAddresses = repo.validated;
        }
        akita::Address defaultAddress;
        std::set<akita::Address> internalAddresses;
        std::set<akita::Address> validatedAddresses;
    };

    struct ChildAccountInformation {
        std::shared_ptr<akita::FeaturedAccount> account;
        DisplayNameStruct displayName;
        AddressRepositoryStruct addresses;

        ChildAccountInformation(std::shared_ptr<akita::FeaturedAccount> acc)
            : account(acc),
              displayName(DisplayNameStruct(acc->displayName)),
              addresses(AddressRepositoryStruct(*acc))
        { }
    };

    AuthResultStruct(akita::BlackBox::FeaturedAccountPtr&& acc, bool needReset, time_t age)
            : addresses(AddressRepositoryStruct(*acc)),
            needReset(needReset),
            attributes(acc->requestAttributes),
            karma(acc->karma),
            aliases(acc->aliases),
            displayName(DisplayNameStruct(acc->displayName)),
            account(std::move(acc)),
            age(age),
            userTicket(account->userTicket)
    {
        boost::copy(account->services, std::back_inserter(services));

        const akita::FeaturedAccount::AccountArray& accs = account->childAccounts;
        std::transform(accs.begin(), accs.end(), std::back_inserter(childAccounts),
                       [] (const auto& acc) { return ChildAccountInformation(acc); });
    }
    AddressRepositoryStruct addresses;
    bool needReset;
    std::vector<akita::ServiceId> services;
    RequestAttributes attributes;
    akita::UserKarma karma;
    akita::FeaturedAccount::Aliases aliases;
    DisplayNameStruct displayName;
    akita::BlackBox::FeaturedAccountPtr account;
    time_t age;
    std::vector<ChildAccountInformation> childAccounts;
    std::string ticket;
    std::string userTicket;
};

inline std::string reflectAuth(akita::BlackBox::FeaturedAccountPtr&& acc, bool needReset, time_t age) {
    AuthResultStruct res(std::move(acc), needReset, age);
    return yamail::data::serialization::toJson(res, "account_information").str();
}

}
}

YREFLECTION_ADAPT_ADT(TzInfo,
        YREFLECTION_ROPROPERTY(std::string, timezone)
        YREFLECTION_ROPROPERTY(int, offset))

BOOST_FUSION_ADAPT_STRUCT( akita::UserKarma,
        (std::string, value)
        (std::string, status) )

BOOST_FUSION_ADAPT_STRUCT( akita::Attributes,
        (bool, haveOrganizationName)
        (bool, haveYaplus)
        (std::string, securityLevel) )

YREFLECTION_ADAPT_ADT(akita::FeaturedAccount,
        YREFLECTION_ROMEMBER(std::string, mailDataBase)
        YREFLECTION_ROMEMBER(std::string, serviceUserId)
        YREFLECTION_ROMEMBER(std::string, firstName)
        YREFLECTION_ROMEMBER(std::string, lastName)
        YREFLECTION_ROMEMBER(std::string, language)
        YREFLECTION_ROMEMBER(std::string, country)
        YREFLECTION_ROMEMBER(TzInfo, timeZone)
        YREFLECTION_ROMEMBER(std::string, birthdayDate)
        YREFLECTION_ROMEMBER(std::time_t, registrationDate)
        YREFLECTION_ROMEMBER(std::string, account)
        YREFLECTION_ROMEMBER(std::string, login)
        YREFLECTION_ROMEMBER(std::string, domain)
        YREFLECTION_ROMEMBER(akita::UserKarma, karma)
        YREFLECTION_ROMEMBER(akita::FeaturedAccount::Aliases, aliases)
        YREFLECTION_ROMEMBER(std::string, composeCheck)
        YREFLECTION_ROMEMBER(std::string, userId)
        YREFLECTION_ROMEMBER(std::string, bbConnectionId)
        YREFLECTION_ROMEMBER(std::string, bbLoginId)
        YREFLECTION_ROMEMBER(int, kind)
        YREFLECTION_ROMEMBER(akita::Attributes, attributes)
        YREFLECTION_ROMEMBER(std::string, gender)
)

BOOST_FUSION_ADAPT_STRUCT( akita::SocialInfo,
        (std::string, profile)
        (std::string, provider)
        (std::string, target) )

BOOST_FUSION_ADAPT_STRUCT(akita::reflection::AuthResultStruct::DisplayNameStruct,
        (std::string, name)
        (bool, social)
        (akita::SocialInfo, socialInfo)
        (std::string, avatar) )

BOOST_FUSION_ADAPT_STRUCT(akita::reflection::AuthResultStruct::AddressRepositoryStruct,
        (akita::Address, defaultAddress)
        (std::set<akita::Address>, internalAddresses)
        (std::set<akita::Address>, validatedAddresses) )

BOOST_FUSION_ADAPT_STRUCT(akita::reflection::AuthResultStruct::ChildAccountInformation,
                         (std::shared_ptr<akita::FeaturedAccount>, account)
                         (akita::reflection::AuthResultStruct::DisplayNameStruct, displayName)
                         (akita::reflection::AuthResultStruct::AddressRepositoryStruct, addresses)
                         )

BOOST_FUSION_ADAPT_STRUCT(akita::reflection::AuthResultStruct,
        (akita::BlackBox::FeaturedAccountPtr, account)
        (akita::reflection::AuthResultStruct::AddressRepositoryStruct, addresses)
        (bool, needReset)
        (akita::RequestAttributes, attributes)
        (std::vector<akita::ServiceId>, services)
        (akita::reflection::AuthResultStruct::DisplayNameStruct, displayName)
        (time_t, age)
        (std::vector<akita::reflection::AuthResultStruct::ChildAccountInformation>, childAccounts)
        (std::string, ticket)
        (std::string, userTicket)
        )
